#!/usr/bin/perl
#
# Monitorix - A lightweight system monitoring tool.
# Copyright (C) 2005-2011 by Jordi Sanfeliu <jordi@fibranet.cat>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

no strict "vars";
no warnings "once";
use CGI qw/:standard/;
use POSIX;
use RRDs;

# load the path of the configuration file
open(IN, "< monitorix.conf.path");
my $opt_config = <IN>;
chomp($opt_config);
close(IN);
if(-f $opt_config) {
	require $opt_config;
} elsif(-f "/etc/monitorix.conf") {
	require "/etc/monitorix.conf";
} elsif(-f "/usr/local/etc/monitorix.conf") {
	require "/usr/local/etc/monitorix.conf";
} else {
	print("Content-Type: text/html\n");
	print("\n");
	print("Monitorix configuration file '$opt_config' not found!\n");
	print("Other possible locations also failed.\n");
	exit(1);
}

our $URL = $ENV{HTTPS} ? "https://" . $ENV{HTTP_HOST} : "http://" . $ENV{HTTP_HOST};
if(!($HOSTNAME)) {
	$HOSTNAME = $ENV{SERVER_NAME};
	if(!($HOSTNAME)) {	# called from the command line
		$HOSTNAME = "127.0.0.1";
		$URL = "http://127.0.0.1";
	}
}
$URL .= $BASE_URL . "/";

my ($os) = uname();
my $mode = defined(param('mode')) ? param('mode') : '';
my $graph = param('graph');
my $when = param('when');
my $color = param('color');
my $val = defined(param('val')) ? param('val') : '';
my $silent = defined(param('silent')) ? param('silent') : '';
if($mode ne "localhost") {
	($mode, $val)  = split(/\./, $mode);
}

our ($res, $tc, $tb, $ts);

($res, $tc, $tb, $ts) = (3600, 'h', 24, 1) if $when eq "day";
($res, $tc, $tb, $ts) = (108000, 'd', 7, 1) if $when eq "week";
($res, $tc, $tb, $ts) = (216000, 'd', 30, 1) if $when eq "month";
($res, $tc, $tb, $ts) = (5184000, 'd', 365, 1) if $when eq "year";

# Default colors (white theme)
our @graph_colors;
our $warning_color = "--color=CANVAS#880000";
our $bg_color = "#FFFFFF";
our $fg_color = "#000000";
our $title_bg_color = "#777777";
our $title_fg_color = "#CCCC00";
our $graph_bg_color = "#CCCCCC";

if($color) {
	if($color eq "black") {
		push(@graph_colors, "--color=CANVAS" . $BLACK{canvas});
		push(@graph_colors, "--color=BACK" . $BLACK{back});
		push(@graph_colors, "--color=FONT" . $BLACK{font});
		push(@graph_colors, "--color=MGRID" . $BLACK{mgrid});
		push(@graph_colors, "--color=GRID" . $BLACK{grid});
		push(@graph_colors, "--color=FRAME" . $BLACK{frame});
		push(@graph_colors, "--color=ARROW" . $BLACK{arrow});
		push(@graph_colors, "--color=SHADEA" . $BLACK{shadea});
		push(@graph_colors, "--color=SHADEB" . $BLACK{shadeb});
		$bg_color = $BLACK{main_bg};
		$fg_color = $BLACK{main_fg};
		$title_bg_color = $BLACK{title_bg};
		$title_fg_color = $BLACK{title_fg};
		$graph_bg_color = $BLACK{graph_bg};
	}
}

my @VERSION12;
my @VERSION12_small;
if($RRDs::VERSION > 1.2) {
	push(@VERSION12, "--slope-mode");
	push(@VERSION12, "--font=LEGEND:7:");
	push(@VERSION12, "--font=TITLE:9:");
	push(@VERSION12, "--font=UNIT:8:");
	if($RRDs::VERSION >= 1.3) {
		push(@VERSION12, "--font=DEFAULT:0:Mono");
	}
	if($when eq "day") {
		push(@VERSION12, "--x-grid=HOUR:1:HOUR:6:HOUR:6:0:%R");
	}
	push(@VERSION12_small, "--font=TITLE:8:");
	push(@VERSION12_small, "--font=UNIT:7:");
	if($RRDs::VERSION >= 1.3) {
		push(@VERSION12_small, "--font=DEFAULT:0:Mono");
	}
} else {
	undef(@VERSION12);
	undef(@VERSION12_small);
}

my $u = "";
if($silent eq "yes" || $silent eq "imagetag") {
	$fg_color = "#000000";	# visible color for text mode
	$u = "_";
}
if($silent eq "imagetagbig") {
	$fg_color = "#000000";	# visible color for text mode
	$u = "";
}


sub get_uptime {
	my $str;
	my $uptime;

	if($os eq "Linux") {
		open(IN, "/proc/uptime");
		($uptime, undef) = split(' ', <IN>);
		close(IN);
	} elsif($os eq "FreeBSD") {
		open(IN, "sysctl -n kern.boottime |");
		(undef, undef, undef, $uptime) = split(' ', <IN>);
		close(IN);
		$uptime =~ s/,//;
		$uptime = time - int($uptime);
	}

	my ($d, $h, $m);
	my ($d_string, $h_string, $m_string);

	$d = int($uptime / (60 * 60 * 24));
	$h = int($uptime / (60 * 60)) % 24;
	$m = int($uptime / 60) % 60;

	$d_string = $d ? sprintf("%d days,", $d) : "";
	$h_string = $h ? sprintf("%d", $h) : "";
	$m_string = $h ? sprintf("%sh %dm", $h, $m) : sprintf("%d min", $m);

	return "$d_string $m_string";
}

# SYSTEM graph
# ----------------------------------------------------------------------------
sub system {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @riglim;
	my @tmp;
	my @tmpz;
	my $n;
	my $err;

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	if($os eq "Linux") {
		$MEMORY = `grep -w MemTotal: /proc/meminfo | awk '{print \$2}'`;
		chomp($MEMORY);
	} elsif($os eq "FreeBSD") {
		$MEMORY = `/sbin/sysctl -n hw.physmem`;	# in bytes
		chomp($MEMORY);
		$MEMORY = int($MEMORY / 1024);		# in KB
	}
	$MEMORY = int($MEMORY / 1024);			# in MB

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$SYSTEM_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		print("ERROR: while fetching $SYSTEM_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		print("       CPU load average    Memory usage in MB     Processes\n");
		print("Time   1min  5min 15min    Used  Cached Buffers   Total   Run\n");
		print("------------------------------------------------------------- \n");
		my $line;
		my @row;
		my $time;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			my ($load1, $load5, $load15, $nproc, $npslp, $nprun, $mtotl, $buff, $cach, $free) = @$line;
			$buff /= 1024;
			$cach /= 1024;
			$free /= 1024;
			@row = ($load1, $load5, $load15, $MEMORY - $free, $cach, $buff, $nproc, $nprun);
			$time = $time - (1 / $ts);
			printf(" %2d$tc   %4.1f  %4.1f  %4.1f  %6d  %6d  %6d   %5d %5d \n", $time, @row);
		}
		print("\n");
		print(" system uptime: " . get_uptime() . "\n");
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}
	if($SYSTEM1_RIGID eq 1) {
		push(@riglim, "--upper-limit=$SYSTEM1_LIMIT");
	} else {
		if($SYSTEM1_RIGID eq 2) {
			push(@riglim, "--upper-limit=$SYSTEM1_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	my $UPTIME = get_uptime();
	my $UPTIMELINE;
	if($RRDs::VERSION > 1.2) {
		$UPTIMELINE = "COMMENT:system uptime\\: " . $UPTIME . "\\c";
	} else {
		$UPTIMELINE = "COMMENT:system uptime: " . $UPTIME . "\\c";
	}

	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}
	push(@tmp, "AREA:load1#4444EE: 1 min average");
	push(@tmp, "GPRINT:load1:LAST:  Current\\: %4.2lf");
	push(@tmp, "GPRINT:load1:AVERAGE:   Average\\: %4.2lf");
	push(@tmp, "GPRINT:load1:MIN:   Min\\: %4.2lf");
	push(@tmp, "GPRINT:load1:MAX:   Max\\: %4.2lf\\n");
	push(@tmp, "LINE1:load1#0000EE");
	push(@tmp, "LINE1:load5#EEEE00: 5 min average");
	push(@tmp, "GPRINT:load5:LAST:  Current\\: %4.2lf");
	push(@tmp, "GPRINT:load5:AVERAGE:   Average\\: %4.2lf");
	push(@tmp, "GPRINT:load5:MIN:   Min\\: %4.2lf");
	push(@tmp, "GPRINT:load5:MAX:   Max\\: %4.2lf\\n");
	push(@tmp, "LINE1:load15#00EEEE:15 min average");
	push(@tmp, "GPRINT:load15:LAST:  Current\\: %4.2lf");
	push(@tmp, "GPRINT:load15:AVERAGE:   Average\\: %4.2lf");
	push(@tmp, "GPRINT:load15:MIN:   Min\\: %4.2lf");
	push(@tmp, "GPRINT:load15:MAX:   Max\\: %4.2lf\\n");
	push(@tmpz, "AREA:load1#4444EE: 1 min average");
	push(@tmpz, "LINE1:load1#0000EE");
	push(@tmpz, "LINE1:load5#EEEE00: 5 min average");
	push(@tmpz, "LINE1:load15#00EEEE:15 min average");
	if($os eq "FreeBSD") {
		push(@tmp, "COMMENT: \\n");
	}
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=System load average  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Load average",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:load1=$SYSTEM_RRD:system_load1:AVERAGE",
		"DEF:load5=$SYSTEM_RRD:system_load5:AVERAGE",
		"DEF:load15=$SYSTEM_RRD:system_load15:AVERAGE",
		@tmp,
		"COMMENT: \\n",
		$UPTIMELINE);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=System load average  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Load average",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:load1=$SYSTEM_RRD:system_load1:AVERAGE",
			"DEF:load5=$SYSTEM_RRD:system_load5:AVERAGE",
			"DEF:load15=$SYSTEM_RRD:system_load15:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /system1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@riglim);
	if($SYSTEM2_RIGID eq 1) {
		push(@riglim, "--upper-limit=$SYSTEM2_LIMIT");
	} else {
		if($SYSTEM2_RIGID eq 2) {
			push(@riglim, "--upper-limit=$SYSTEM2_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:npslp#44AAEE:Sleeping");
	push(@tmp, "AREA:nprun#EE4444:Running");
	push(@tmp, "LINE1:nprun#EE0000");
	push(@tmp, "LINE1:npslp#00EEEE");
	push(@tmp, "LINE1:nproc#EEEE00:Processes");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=Active processes  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Processes",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:nproc=$SYSTEM_RRD:system_nproc:AVERAGE",
		"DEF:npslp=$SYSTEM_RRD:system_npslp:AVERAGE",
		"DEF:nprun=$SYSTEM_RRD:system_nprun:AVERAGE",
		@tmp,
		"COMMENT: \\n");
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=Active processes  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Processes",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:nproc=$SYSTEM_RRD:system_nproc:AVERAGE",
			"DEF:npslp=$SYSTEM_RRD:system_npslp:AVERAGE",
			"DEF:nprun=$SYSTEM_RRD:system_nprun:AVERAGE",
			@tmp);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /system2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:m_mused#EE4444:Used");
	push(@tmp, "AREA:m_mcach#44EE44:Cached");
	push(@tmp, "AREA:m_mbuff#CCCCCC:Buffers");
	if($os eq "FreeBSD") {
		push(@tmp, "AREA:m_macti#EEEE44:Active");
		push(@tmp, "AREA:m_minac#4444EE:Inactive");
		push(@tmp, "LINE1:m_minac#0000EE");
		push(@tmp, "LINE1:m_macti#EEEE00");
	}
	push(@tmp, "LINE1:m_mbuff#888888");
	push(@tmp, "LINE1:m_mcach#00EE00");
	push(@tmp, "LINE1:m_mused#EE0000");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=Memory allocation (${MEMORY}MB)  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Megabytes",
		"--width=$width",
		"--height=$height",
		"--upper-limit=$MEMORY",
		"--lower-limit=0",
		"--base=1024",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:mtotl=$SYSTEM_RRD:system_mtotl:AVERAGE",
		"DEF:mbuff=$SYSTEM_RRD:system_mbuff:AVERAGE",
		"DEF:mcach=$SYSTEM_RRD:system_mcach:AVERAGE",
		"DEF:mfree=$SYSTEM_RRD:system_mfree:AVERAGE",
		"DEF:macti=$SYSTEM_RRD:system_macti:AVERAGE",
		"DEF:minac=$SYSTEM_RRD:system_minac:AVERAGE",
		"CDEF:m_mtotl=mtotl,1024,/",
		"CDEF:m_mbuff=mbuff,1024,/",
		"CDEF:m_mcach=mcach,1024,/",
		"CDEF:m_mused=m_mtotl,mfree,1024,/,-",
		"CDEF:m_macti=macti,1024,/",
		"CDEF:m_minac=minac,1024,/",
		@tmp,
		"COMMENT: \\n");
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=Memory allocation (${MEMORY}MB)  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Megabytes",
			"--width=$width",
			"--height=$height",
			"--upper-limit=$MEMORY",
			"--lower-limit=0",
			"--base=1024",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:mtotl=$SYSTEM_RRD:system_mtotl:AVERAGE",
			"DEF:mbuff=$SYSTEM_RRD:system_mbuff:AVERAGE",
			"DEF:mcach=$SYSTEM_RRD:system_mcach:AVERAGE",
			"DEF:mfree=$SYSTEM_RRD:system_mfree:AVERAGE",
			"DEF:macti=$SYSTEM_RRD:system_macti:AVERAGE",
			"DEF:minac=$SYSTEM_RRD:system_minac:AVERAGE",
			"CDEF:m_mtotl=mtotl,1024,/",
			"CDEF:m_mbuff=mbuff,1024,/",
			"CDEF:m_mcach=mcach,1024,/",
			"CDEF:m_mused=m_mtotl,mfree,1024,/,-",
			"CDEF:m_macti=macti,1024,/",
			"CDEF:m_minac=minac,1024,/",
			@tmp);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /system3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# KERN graph
# ----------------------------------------------------------------------------
sub kern {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @riglim;
	my @tmp;
	my @tmpz;
	my $vlabel;
	my $n;
	my $err;

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$KERN_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		print("ERROR: while fetching $KERN_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		print("       Kernel usage                                                            VFS usage\n");
		print("Time   User   Nice    Sys   Idle   I/Ow    IRQ   sIRQ  Steal  Guest   Ctxt.Sw  dentry   file  inode\n");
		print("---------------------------------------------------------------------------------------------------\n");
		my $line;
		my @row;
		my $time;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			my ($usr, $nic, $sys, $idle, $iow, $irq, $sirq, $steal, $guest, $cs, $dentry, $file, $inode) = @$line;
			@row = ($usr, $nic, $sys, $idle, $iow, $irq, $sirq, $steal, $guest, $cs, $dentry, $file, $inode);
			$time = $time - (1 / $ts);
			printf(" %2d$tc  %4.1f%%  %4.1f%%  %4.1f%%  %4.1f%%  %4.1f%%  %4.1f%%  %4.1f%%  %4.1f%%  %4.1f%%   %6d    %4.1f%%  %4.1f%%  %4.1f%% \n", $time, @row);
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}

	if($KERN1_RIGID eq 1) {
		push(@riglim, "--upper-limit=$KERN1_LIMIT");
	} else {
		if($KERN1_RIGID eq 2) {
			push(@riglim, "--upper-limit=$KERN1_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	if($KERN_GRAPH_MODE eq "R") {
		$vlabel = "Percent (%)";
		if($KERN_DATA{user} eq "Y") {
			push(@tmp, "AREA:user#4444EE:user");
			push(@tmpz, "AREA:user#4444EE:user");
			push(@tmp, "GPRINT:user:LAST:      Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:user:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:user:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:user:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{nice} eq "Y") {
			push(@tmp, "AREA:nice#EEEE44:nice");
			push(@tmpz, "AREA:nice#EEEE44:nice");
			push(@tmp, "GPRINT:nice:LAST:      Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:nice:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:nice:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:nice:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{sys} eq "Y") {
			push(@tmp, "AREA:sys#44EEEE:system");
			push(@tmpz, "AREA:sys#44EEEE:system");
			push(@tmp, "GPRINT:sys:LAST:    Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:sys:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:sys:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:sys:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{iow} eq "Y") {
			push(@tmp, "AREA:iow#EE44EE:I/O wait");
			push(@tmpz, "AREA:iow#EE44EE:I/O wait");
			push(@tmp, "GPRINT:iow:LAST:  Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:iow:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:iow:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:iow:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{irq} eq "Y") {
			push(@tmp, "AREA:irq#888888:IRQ");
			push(@tmpz, "AREA:irq#888888:IRQ");
			push(@tmp, "GPRINT:irq:LAST:       Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:irq:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:irq:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:irq:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{sirq} eq "Y") {
			push(@tmp, "AREA:sirq#E29136:softIRQ");
			push(@tmpz, "AREA:sirq#E29136:softIRQ");
			push(@tmp, "GPRINT:sirq:LAST:   Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:sirq:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:sirq:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:sirq:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{steal} eq "Y") {
			push(@tmp, "AREA:steal#44EE44:steal");
			push(@tmpz, "AREA:steal#44EE44:steal");
			push(@tmp, "GPRINT:steal:LAST:     Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:steal:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:steal:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:steal:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{guest} eq "Y") {
			push(@tmp, "AREA:guest#448844:guest");
			push(@tmpz, "AREA:guest#448844:guest");
			push(@tmp, "GPRINT:guest:LAST:     Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:guest:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:guest:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:guest:MAX:    Max\\: %4.1lf%%\\n");
		}
		push(@tmp, "LINE1:guest#1F881F") unless $KERN_DATA{guest} ne "Y";
		push(@tmpz, "LINE1:guest#1F881F") unless $KERN_DATA{guest} ne "Y";
		push(@tmp, "LINE1:steal#00EE00") unless $KERN_DATA{steal} ne "Y";
		push(@tmpz, "LINE1:steal#00EE00") unless $KERN_DATA{steal} ne "Y";
		push(@tmp, "LINE1:sirq#D86612") unless $KERN_DATA{sirq} ne "Y";
		push(@tmpz, "LINE1:sirq#D86612") unless $KERN_DATA{sirq} ne "Y";
		push(@tmp, "LINE1:irq#CCCCCC") unless $KERN_DATA{irq} ne "Y";
		push(@tmpz, "LINE1:irq#CCCCCC") unless $KERN_DATA{irq} ne "Y";
		push(@tmp, "LINE1:iow#EE00EE") unless $KERN_DATA{iow} ne "Y";
		push(@tmpz, "LINE1:iow#EE00EE") unless $KERN_DATA{iow} ne "Y";
		push(@tmp, "LINE1:sys#00EEEE") unless $KERN_DATA{sys} ne "Y";
		push(@tmpz, "LINE1:sys#00EEEE") unless $KERN_DATA{sys} ne "Y";
		push(@tmp, "LINE1:nice#EEEE00") unless $KERN_DATA{nice} ne "Y";
		push(@tmpz, "LINE1:nice#EEEE00") unless $KERN_DATA{nice} ne "Y";
		push(@tmp, "LINE1:user#0000EE") unless $KERN_DATA{user} ne "Y";
		push(@tmpz, "LINE1:user#0000EE") unless $KERN_DATA{user} ne "Y";
	} else {
		$vlabel = "Stacked Percent (%)";
		push(@tmp, "CDEF:s_nice=user,nice,+");
		push(@tmpz, "CDEF:s_nice=user,nice,+");
		push(@tmp, "CDEF:s_sys=s_nice,sys,+");
		push(@tmpz, "CDEF:s_sys=s_nice,sys,+");
		push(@tmp, "CDEF:s_iow=s_sys,iow,+");
		push(@tmpz, "CDEF:s_iow=s_sys,iow,+");
		push(@tmp, "CDEF:s_irq=s_iow,irq,+");
		push(@tmpz, "CDEF:s_irq=s_iow,irq,+");
		push(@tmp, "CDEF:s_sirq=s_irq,sirq,+");
		push(@tmpz, "CDEF:s_sirq=s_irq,sirq,+");
		push(@tmp, "CDEF:s_steal=s_sirq,steal,+");
		push(@tmpz, "CDEF:s_steal=s_sirq,steal,+");
		push(@tmp, "CDEF:s_guest=s_steal,guest,+");
		push(@tmpz, "CDEF:s_guest=s_steal,guest,+");
		if($KERN_DATA{guest} eq "Y") {
			push(@tmp, "AREA:s_guest#448844:guest");
			push(@tmpz, "AREA:s_guest#448844:guest");
			push(@tmp, "GPRINT:guest:LAST:     Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:guest:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:guest:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:guest:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{steal} eq "Y") {
			push(@tmp, "AREA:s_steal#44EE44:steal");
			push(@tmpz, "AREA:s_steal#44EE44:steal");
			push(@tmp, "GPRINT:steal:LAST:     Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:steal:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:steal:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:steal:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{sirq} eq "Y") {
			push(@tmp, "AREA:s_sirq#E29136:softIRQ");
			push(@tmpz, "AREA:s_sirq#E29136:softIRQ");
			push(@tmp, "GPRINT:sirq:LAST:   Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:sirq:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:sirq:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:sirq:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{irq} eq "Y") {
			push(@tmp, "AREA:s_irq#888888:IRQ");
			push(@tmpz, "AREA:s_irq#888888:IRQ");
			push(@tmp, "GPRINT:irq:LAST:       Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:irq:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:irq:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:irq:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{iow} eq "Y") {
			push(@tmp, "AREA:s_iow#EE44EE:I/O wait");
			push(@tmpz, "AREA:s_iow#EE44EE:I/O wait");
			push(@tmp, "GPRINT:iow:LAST:  Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:iow:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:iow:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:iow:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{sys} eq "Y") {
			push(@tmp, "AREA:s_sys#44EEEE:system");
			push(@tmpz, "AREA:s_sys#44EEEE:system");
			push(@tmp, "GPRINT:sys:LAST:    Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:sys:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:sys:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:sys:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{nice} eq "Y") {
			push(@tmp, "AREA:s_nice#EEEE44:nice");
			push(@tmpz, "AREA:s_nice#EEEE44:nice");
			push(@tmp, "GPRINT:nice:LAST:      Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:nice:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:nice:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:nice:MAX:    Max\\: %4.1lf%%\\n");
		}
		if($KERN_DATA{user} eq "Y") {
			push(@tmp, "AREA:user#4444EE:user");
			push(@tmpz, "AREA:user#4444EE:user");
			push(@tmp, "GPRINT:user:LAST:      Current\\: %4.1lf%%");
			push(@tmp, "GPRINT:user:AVERAGE:    Average\\: %4.1lf%%");
			push(@tmp, "GPRINT:user:MIN:    Min\\: %4.1lf%%");
			push(@tmp, "GPRINT:user:MAX:    Max\\: %4.1lf%%\\n");
		}
		push(@tmp, "LINE1:s_guest#1F881F");
		push(@tmpz, "LINE1:s_guest#1F881F");
		push(@tmp, "LINE1:s_steal#00EE00");
		push(@tmpz, "LINE1:s_steal#00EE00");
		push(@tmp, "LINE1:s_sirq#D86612");
		push(@tmpz, "LINE1:s_sirq#D86612");
		push(@tmp, "LINE1:s_irq#CCCCCC");
		push(@tmpz, "LINE1:s_irq#CCCCCC");
		push(@tmp, "LINE1:s_iow#EE00EE");
		push(@tmpz, "LINE1:s_iow#EE00EE");
		push(@tmp, "LINE1:s_sys#00EEEE");
		push(@tmpz, "LINE1:s_sys#00EEEE");
		push(@tmp, "LINE1:s_nice#EEEE00");
		push(@tmpz, "LINE1:s_nice#EEEE00");
		push(@tmp, "LINE1:user#0000EE");
		push(@tmpz, "LINE1:user#0000EE");
	}

	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Kernel usage  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=$vlabel",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:user=$KERN_RRD:kern_user:AVERAGE",
		"DEF:nice=$KERN_RRD:kern_nice:AVERAGE",
		"DEF:sys=$KERN_RRD:kern_sys:AVERAGE",
		"DEF:iow=$KERN_RRD:kern_iow:AVERAGE",
		"DEF:irq=$KERN_RRD:kern_irq:AVERAGE",
		"DEF:sirq=$KERN_RRD:kern_sirq:AVERAGE",
		"DEF:steal=$KERN_RRD:kern_steal:AVERAGE",
		"DEF:guest=$KERN_RRD:kern_guest:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Kernel usage  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=$vlabel",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:user=$KERN_RRD:kern_user:AVERAGE",
			"DEF:nice=$KERN_RRD:kern_nice:AVERAGE",
			"DEF:sys=$KERN_RRD:kern_sys:AVERAGE",
			"DEF:iow=$KERN_RRD:kern_iow:AVERAGE",
			"DEF:irq=$KERN_RRD:kern_irq:AVERAGE",
			"DEF:sirq=$KERN_RRD:kern_sirq:AVERAGE",
			"DEF:steal=$KERN_RRD:kern_steal:AVERAGE",
			"DEF:guest=$KERN_RRD:kern_guest:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /kern1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:cs#44AAEE:Context switches");
	push(@tmpz, "AREA:cs#44AAEE:Context switches");
	push(@tmp, "GPRINT:cs:LAST:   Current\\: %6.0lf\\n");
	push(@tmp, "LINE1:cs#00EEEE");
	push(@tmpz, "LINE1:cs#00EEEE");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=Context switches  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=CS/sec",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:cs=$KERN_RRD:kern_cs:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=Context switches  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=CS/sec",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:cs=$KERN_RRD:kern_cs:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /kern2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@tmp);
	undef(@tmpz);
	if($os eq "Linux") {
		push(@tmp, "AREA:dentry#4444EE:dentry");
		push(@tmpz, "AREA:dentry#4444EE:dentry");
		push(@tmp, "GPRINT:dentry:LAST:          Current\\:  %4.1lf%%\\n");
	}
	push(@tmp, "AREA:inode#EEEE44:inode");
	push(@tmpz, "AREA:inode#EEEE44:inode");
	push(@tmp, "GPRINT:inode:LAST:           Current\\:  %4.1lf%%\\n");
	push(@tmp, "AREA:file#EE44EE:file");
	push(@tmpz, "AREA:file#EE44EE:file");
	push(@tmp, "GPRINT:file:LAST:            Current\\:  %4.1lf%%\\n");
	push(@tmp, "LINE1:dentry#0000EE");
	push(@tmpz, "LINE1:dentry#0000EE");
	push(@tmp, "LINE1:inode#EEEE00");
	push(@tmpz, "LINE1:inode#EEEE00");
	push(@tmp, "LINE1:file#EE00EE");
	push(@tmpz, "LINE1:file#EE00EE");
	if($os eq "FreeBSD") {
		push(@tmp, "COMMENT: \\n");
	}
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=VFS usage  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Percent (%)",
		"--width=$width",
		"--height=$height",
		"--upper-limit=100",
		"--lower-limit=0",
		"--rigid",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:dentry=$KERN_RRD:kern_dentry:AVERAGE",
		"DEF:file=$KERN_RRD:kern_file:AVERAGE",
		"DEF:inode=$KERN_RRD:kern_inode:AVERAGE",
		@tmp,
		"COMMENT: \\n");
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=VFS usage  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Percent (%)",
			"--width=$width",
			"--height=$height",
			"--upper-limit=100",
			"--lower-limit=0",
			"--rigid",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:dentry=$KERN_RRD:kern_dentry:AVERAGE",
			"DEF:file=$KERN_RRD:kern_file:AVERAGE",
			"DEF:inode=$KERN_RRD:kern_inode:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /kern3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# PROC graph
# ----------------------------------------------------------------------------
sub proc {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @riglim;
	my @PNG;
	my @PNGz;
	my @tmp;
	my @tmpz;
	my $vlabel;
	my $ncpu;
	my $n;
	my $str;
	my $err;

	if($os eq "Linux") {
		$ncpu = `grep -w processor /proc/cpuinfo | tail -1 | awk '{ print \$3 }'`;
		chomp($ncpu);
		$ncpu++;
	} elsif($os eq "FreeBSD") {
		$ncpu = `/sbin/sysctl -n hw.ncpu`;
		chomp($ncpu);
	}
	$ncpu = $ncpu > $PROC_MAX ? $PROC_MAX : $ncpu;
	return 0 unless $ncpu > 1;

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$PROC_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		print("ERROR: while fetching $PROC_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		for($n = 0; $n < $ncpu; $n++) {
			print("       Processor " . sprintf("%3d", $n) . "                                   ");
		}
		print("\nTime");
		for($n = 0; $n < $ncpu; $n++) {
			print("   User  Nice   Sys  Idle  I/Ow   IRQ  sIRQ Steal Guest");
		}
		print(" \n----");
		for($n = 0; $n < $ncpu; $n++) {
			print("-------------------------------------------------------");
		}
		print(" \n");
		my $line;
		my @row;
		my $time;
		my $n2;
		my $from;
		my $to;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			$time = $time - (1 / $ts);
			printf(" %2d$tc ", $time);
			for($n2 = 0; $n2 < $ncpu; $n2++) {
				$from = $n2 * $ncpu;
				$to = $from + $ncpu;
				my ($usr, $nic, $sys, $idle, $iow, $irq, $sirq, $steal, $guest,) = @$line[$from..$to];
				@row = ($usr, $nic, $sys, $idle, $iow, $irq, $sirq, $steal, $guest);
				printf(" %4.1f%% %4.1f%% %4.1f%% %4.1f%% %4.1f%% %4.1f%% %4.1f%% %4.1f%% %4.1f%% ", @row);
			}
			print("\n");
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	for($n = 0; $n < $ncpu; $n++) {
		$str = $u . $myself . $n . "." . $when . ".png";
		push(@PNG, $str);
		unlink("$PNG_DIR" . $str);
		if($ENABLE_ZOOM eq "Y") {
			$str = $u . $myself . $n . "z." . $when . ".png";
			push(@PNGz, $str);
			unlink("$PNG_DIR" . $str);
		}
	}

	$n = 0;
	if($PROC_RIGID eq 1) {
		push(@riglim, "--upper-limit=$PROC_LIMIT");
	} else {
		if($PROC_RIGID eq 2) {
			push(@riglim, "--upper-limit=$PROC_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	while($n < $ncpu) {
		if($title) {
			if($n == 0) {
				graph_header($title, $PROC_PER_ROW);
			}
			print("    <tr>\n");
		}
		for($n2 = 0; $n2 < $PROC_PER_ROW; $n2++) {
			last unless $n < $ncpu;
			if($title) {
				print("    <td bgcolor='" . $title_bg_color . "'>\n");
			}
			undef(@tmp);
			undef(@tmpz);
			if($KERN_GRAPH_MODE eq "R") {
				$vlabel = "Percent (%)";
				if($KERN_DATA{user} eq "Y") {
					push(@tmp, "AREA:user#4444EE:user");
					push(@tmpz, "AREA:user#4444EE:user");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:user:LAST:    Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:user:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:user:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:user:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{nice} eq "Y") {
					push(@tmp, "AREA:nice#EEEE44:nice");
					push(@tmpz, "AREA:nice#EEEE44:nice");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:nice:LAST:    Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:nice:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:nice:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:nice:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{sys} eq "Y") {
					push(@tmp, "AREA:sys#44EEEE:system");
					push(@tmpz, "AREA:sys#44EEEE:system");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:sys:LAST:  Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:sys:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:sys:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:sys:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{iow} eq "Y") {
					push(@tmp, "AREA:iow#EE44EE:I/O wait");
					push(@tmpz, "AREA:iow#EE44EE:I/O wait");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:iow:LAST:Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:iow:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:iow:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:iow:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{irq} eq "Y") {
					push(@tmp, "AREA:irq#888888:IRQ");
					push(@tmpz, "AREA:irq#888888:IRQ");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:irq:LAST:     Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:irq:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:irq:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:irq:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{sirq} eq "Y") {
					push(@tmp, "AREA:sirq#E29136:softIRQ");
					push(@tmpz, "AREA:sirq#E29136:softIRQ");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:sirq:LAST: Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:sirq:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:sirq:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:sirq:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{steal} eq "Y") {
					push(@tmp, "AREA:steal#44EE44:steal");
					push(@tmpz, "AREA:steal#44EE44:steal");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:steal:LAST:   Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:steal:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:steal:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:steal:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{guest} eq "Y") {
					push(@tmp, "AREA:guest#448844:guest");
					push(@tmpz, "AREA:guest#448844:guest");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:guest:LAST:   Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:guest:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:guest:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:guest:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				push(@tmp, "LINE1:guest#1F881F") unless $KERN_DATA{guest} ne "Y";
				push(@tmpz, "LINE1:guest#1F881F") unless $KERN_DATA{guest} ne "Y";
				push(@tmp, "LINE1:steal#00EE00") unless $KERN_DATA{steal} ne "Y";
				push(@tmpz, "LINE1:steal#00EE00") unless $KERN_DATA{steal} ne "Y";
				push(@tmp, "LINE1:sirq#D86612") unless $KERN_DATA{sirq} ne "Y";
				push(@tmpz, "LINE1:sirq#D86612") unless $KERN_DATA{sirq} ne "Y";
				push(@tmp, "LINE1:irq#CCCCCC") unless $KERN_DATA{irq} ne "Y";
				push(@tmpz, "LINE1:irq#CCCCCC") unless $KERN_DATA{irq} ne "Y";
				push(@tmp, "LINE1:iow#EE00EE") unless $KERN_DATA{iow} ne "Y";
				push(@tmpz, "LINE1:iow#EE00EE") unless $KERN_DATA{iow} ne "Y";
				push(@tmp, "LINE1:sys#00EEEE") unless $KERN_DATA{sys} ne "Y";
				push(@tmpz, "LINE1:sys#00EEEE") unless $KERN_DATA{sys} ne "Y";
				push(@tmp, "LINE1:nice#EEEE00") unless $KERN_DATA{nice} ne "Y";
				push(@tmpz, "LINE1:nice#EEEE00") unless $KERN_DATA{nice} ne "Y";
				push(@tmp, "LINE1:user#0000EE") unless $KERN_DATA{user} ne "Y";
				push(@tmpz, "LINE1:user#0000EE") unless $KERN_DATA{user} ne "Y";
			} else {
				$vlabel = "Stacked Percent (%)";
				push(@tmp, "CDEF:s_nice=user,nice,+");
				push(@tmpz, "CDEF:s_nice=user,nice,+");
				push(@tmp, "CDEF:s_sys=s_nice,sys,+");
				push(@tmpz, "CDEF:s_sys=s_nice,sys,+");
				push(@tmp, "CDEF:s_iow=s_sys,iow,+");
				push(@tmpz, "CDEF:s_iow=s_sys,iow,+");
				push(@tmp, "CDEF:s_irq=s_iow,irq,+");
				push(@tmpz, "CDEF:s_irq=s_iow,irq,+");
				push(@tmp, "CDEF:s_sirq=s_irq,sirq,+");
				push(@tmpz, "CDEF:s_sirq=s_irq,sirq,+");
				push(@tmp, "CDEF:s_steal=s_sirq,steal,+");
				push(@tmpz, "CDEF:s_steal=s_sirq,steal,+");
				push(@tmp, "CDEF:s_guest=s_steal,guest,+");
				push(@tmpz, "CDEF:s_guest=s_steal,guest,+");
				if($KERN_DATA{guest} eq "Y") {
					push(@tmp, "AREA:s_guest#E29136:guest");
					push(@tmpz, "AREA:s_guest#E29136:guest");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:guest:LAST:   Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:guest:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:guest:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:guest:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{steal} eq "Y") {
					push(@tmp, "AREA:s_steal#888888:steal");
					push(@tmpz, "AREA:s_steal#888888:steal");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:steal:LAST:   Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:steal:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:steal:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:steal:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{sirq} eq "Y") {
					push(@tmp, "AREA:s_sirq#448844:softIRQ");
					push(@tmpz, "AREA:s_sirq#448844:softIRQ");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:sirq:LAST: Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:sirq:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:sirq:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:sirq:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{irq} eq "Y") {
					push(@tmp, "AREA:s_irq#44EE44:IRQ");
					push(@tmpz, "AREA:s_irq#44EE44:IRQ");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:irq:LAST:     Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:irq:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:irq:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:irq:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{iow} eq "Y") {
					push(@tmp, "AREA:s_iow#EE44EE:I/O wait");
					push(@tmpz, "AREA:s_iow#EE44EE:I/O wait");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:iow:LAST:Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:iow:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:iow:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:iow:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{sys} eq "Y") {
					push(@tmp, "AREA:s_sys#44EEEE:system");
					push(@tmpz, "AREA:s_sys#44EEEE:system");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:sys:LAST:  Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:sys:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:sys:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:sys:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{nice} eq "Y") {
					push(@tmp, "AREA:s_nice#EEEE44:nice");
					push(@tmpz, "AREA:s_nice#EEEE44:nice");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:nice:LAST:    Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:nice:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:nice:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:nice:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				if($KERN_DATA{user} eq "Y") {
					push(@tmp, "AREA:user#4444EE:user");
					push(@tmpz, "AREA:user#4444EE:user");
					if($PROC_DATA eq "Y") {
						push(@tmp, "GPRINT:user:LAST:    Cur\\: %4.1lf%%");
						push(@tmp, "GPRINT:user:AVERAGE:  Avg\\: %4.1lf%%");
						push(@tmp, "GPRINT:user:MIN:  Min\\: %4.1lf%%");
						push(@tmp, "GPRINT:user:MAX:  Max\\: %4.1lf%%\\n");
					}
				}
				push(@tmp, "LINE1:s_guest#D86612");
				push(@tmpz, "LINE1:s_guest#D86612");
				push(@tmp, "LINE1:s_steal#CCCCCC");
				push(@tmpz, "LINE1:s_steal#CCCCCC");
				push(@tmp, "LINE1:s_sirq#1F881F");
				push(@tmpz, "LINE1:s_sirq#1F881F");
				push(@tmp, "LINE1:s_irq#00EE00");
				push(@tmpz, "LINE1:s_irq#00EE00");
				push(@tmp, "LINE1:s_iow#EE00EE");
				push(@tmpz, "LINE1:s_iow#EE00EE");
				push(@tmp, "LINE1:s_sys#00EEEE");
				push(@tmpz, "LINE1:s_sys#00EEEE");
				push(@tmp, "LINE1:s_nice#EEEE00");
				push(@tmpz, "LINE1:s_nice#EEEE00");
				push(@tmp, "LINE1:user#0000EE");
				push(@tmpz, "LINE1:user#0000EE");
			}
			($width, $height) = split('x', $GRAPH_SIZE{$PROC_SIZE});
			RRDs::graph("$PNG_DIR" . "$PNG[$n]",
				"--title=Processor $n  (1$when)",
				"--start=-1$when",
				"--imgformat=PNG",
				"--vertical-label=$vlabel",
				"--width=$width",
				"--height=$height",
				@riglim,
				"--lower-limit=0",
				@VERSION12,
				@VERSION12_small,
				@graph_colors,
				"DEF:user=$PROC_RRD:proc" . $n . "_user:AVERAGE",
				"DEF:nice=$PROC_RRD:proc" . $n . "_nice:AVERAGE",
				"DEF:sys=$PROC_RRD:proc" . $n . "_sys:AVERAGE",
				"DEF:iow=$PROC_RRD:proc" . $n . "_iow:AVERAGE",
				"DEF:irq=$PROC_RRD:proc" . $n . "_irq:AVERAGE",
				"DEF:sirq=$PROC_RRD:proc" . $n . "_sirq:AVERAGE",
				"DEF:steal=$PROC_RRD:proc" . $n . "_steal:AVERAGE",
				"DEF:guest=$PROC_RRD:proc" . $n . "_guest:AVERAGE",
				@tmp);
			$err = RRDs::error;
			print("ERROR: while graphing $PNG_DIR" . "$PNG[$n]: $err\n") if $err;
			if($ENABLE_ZOOM eq "Y") {
				($width, $height) = split('x', $GRAPH_SIZE{zoom});
				RRDs::graph("$PNG_DIR" . "$PNGz[$n]",
					"--title=Processor $n  (1$when)",
					"--start=-1$when",
					"--imgformat=PNG",
					"--vertical-label=$vlabel",
					"--width=$width",
					"--height=$height",
					@riglim,
					"--lower-limit=0",
					@VERSION12,
					@VERSION12_small,
					@graph_colors,
					"DEF:user=$PROC_RRD:proc" . $n . "_user:AVERAGE",
					"DEF:nice=$PROC_RRD:proc" . $n . "_nice:AVERAGE",
					"DEF:sys=$PROC_RRD:proc" . $n . "_sys:AVERAGE",
					"DEF:iow=$PROC_RRD:proc" . $n . "_iow:AVERAGE",
					"DEF:irq=$PROC_RRD:proc" . $n . "_irq:AVERAGE",
					"DEF:sirq=$PROC_RRD:proc" . $n . "_sirq:AVERAGE",
					"DEF:steal=$PROC_RRD:proc" . $n . "_steal:AVERAGE",
					"DEF:guest=$PROC_RRD:proc" . $n . "_guest:AVERAGE",
					@tmpz);
				$err = RRDs::error;
				print("ERROR: while graphing $$PNG_DIR" . "$PNGz[$n]: $err\n") if $err;
			}
			if($title || ($silent =~ /imagetag/ && $graph =~ /proc$n/)) {
				if($ENABLE_ZOOM eq "Y") {
					print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNGz[$n] . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG[$n] . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG[$n] . "'>\n");
				}
			}
			if($title) {
				print("    </td>\n");
			}
			$n++;
		}
		if($title) {
			print("    </tr>\n");
		}
	}
	if($title) {
		graph_footer();
	}
	return 1;
}

# HPTEMP graph
# ----------------------------------------------------------------------------
sub hptemp {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @tmp;
	my @tmpz;
	my $n;
	my $id;
	my $str;
	my $err;
	my @LC = (
		"#FFA500",
		"#44EEEE",
		"#44EE44",
		"#4444EE",
		"#448844",
		"#EE4444",
		"#EE44EE",
		"#EEEE44",
	);

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	open(IN, "monitorix.hplog");
	my @hplog = <IN>;
	close(IN);

	if(!scalar(@hplog)) {
		print("WARNING: 'hplog' command output is empty.");
	}
	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$HPTEMP_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		my $str;
		my $line1;
		my $line2;
		print("ERROR: while fetching $HPTEMP_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		foreach my $t (@HPTEMP_1, @HPTEMP_2, @HPTEMP_3) {
			$id = sprintf("%2d", $t);
			for($n = 0; $n < scalar(@hplog); $n++) {
				$_ = $hplog[$n];
				if(/^$id  /) {
					$str = substr($_, 17, 8);
					$str = sprintf("%8s", $str);
					$line1 .= "  ";
					$line1 .= $str;
					$line2 .= "----------";
				}
			}
		}
		print("Time $line1 \n");
		print("-----$line2\n");
		my $line;
		my @row;
		my $time;
		my $n2;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			$time = $time - (1 / $ts);
			printf(" %2d$tc ", $time);
			undef($line1);
			undef(@row);
			for($n2 = 0; $n2 < scalar(@HPTEMP_1); $n2++) {
				my $temp = @$line[$n2];
				push(@row, $temp);
				$line1 .= " %8.0f ";
			}
			for($n2 = 0; $n2 < scalar(@HPTEMP_2); $n2++) {
				my $temp = @$line[8 + $n2];
				push(@row, $temp);
				$line1 .= " %8.0f ";
			}
			for($n2 = 0; $n2 < scalar(@HPTEMP_3); $n2++) {
				my $temp = @$line[8 + 3 + $n2];
				push(@row, $temp);
				$line1 .= " %8.0f ";
			}
			print(sprintf($line1, @row));
			print("\n");
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}
	for($n = 0; $n < 8; $n++) {
		if($HPTEMP_1[$n]) {
			foreach $_ (@hplog) {
				$id = sprintf("%2d", $HPTEMP_1[$n]);
				if(/^$id  /) {
					$str = substr($_, 17, 8);
					$str = sprintf("%-20s", $str);
					push(@tmp, "LINE1:temp" . $n . $LC[$n] . ":$str");
					push(@tmp, "GPRINT:temp" . $n . ":LAST:Current\\: %2.0lf");
					push(@tmp, "GPRINT:temp" . $n . ":AVERAGE:   Average\\: %2.0lf");
					push(@tmp, "GPRINT:temp" . $n . ":MIN:   Min\\: %2.0lf");
					push(@tmp, "GPRINT:temp" . $n . ":MAX:   Max\\: %2.0lf\\n");
					$str =~ s/\s+$//;
					push(@tmpz, "LINE2:temp" . $n . $LC[$n] . ":$str");
					last;
				}
			}
		} else {
			push(@tmp, "COMMENT: \\n");
		}
	}

	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Temperatures  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Celsius",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:temp0=$HPTEMP_RRD:hptemp1_1:AVERAGE",
		"DEF:temp1=$HPTEMP_RRD:hptemp1_2:AVERAGE",
		"DEF:temp2=$HPTEMP_RRD:hptemp1_3:AVERAGE",
		"DEF:temp3=$HPTEMP_RRD:hptemp1_4:AVERAGE",
		"DEF:temp4=$HPTEMP_RRD:hptemp1_5:AVERAGE",
		"DEF:temp5=$HPTEMP_RRD:hptemp1_6:AVERAGE",
		"DEF:temp6=$HPTEMP_RRD:hptemp1_7:AVERAGE",
		"DEF:temp7=$HPTEMP_RRD:hptemp1_8:AVERAGE",
		@tmp,
		"COMMENT: \\n");
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Temperatures  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Celsius",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:temp0=$HPTEMP_RRD:hptemp1_1:AVERAGE",
			"DEF:temp1=$HPTEMP_RRD:hptemp1_2:AVERAGE",
			"DEF:temp2=$HPTEMP_RRD:hptemp1_3:AVERAGE",
			"DEF:temp3=$HPTEMP_RRD:hptemp1_4:AVERAGE",
			"DEF:temp4=$HPTEMP_RRD:hptemp1_5:AVERAGE",
			"DEF:temp5=$HPTEMP_RRD:hptemp1_6:AVERAGE",
			"DEF:temp6=$HPTEMP_RRD:hptemp1_7:AVERAGE",
			"DEF:temp7=$HPTEMP_RRD:hptemp1_8:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /hptemp1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@tmp);
	undef(@tmpz);
	for($n = 0; $n < 6; $n++) {
		if($HPTEMP_2[$n]) {
			foreach $_ (@hplog) {
				$id = sprintf("%2d", $HPTEMP_2[$n]);
				if(/^$id  /) {
					$str = substr($_, 17, 8);
					$str = sprintf("%-8s", $str);
					push(@tmp, "LINE1:temp" . $n . $LC[$n] . ":$str");
					push(@tmp, "GPRINT:temp" . $n . ":LAST:\\: %2.0lf");
					if(!(($n + 1) % 2)) {
						push(@tmp, "COMMENT: \\n");
					} else {
						push(@tmp, "COMMENT:    ");
					}
					$str =~ s/\s+$//;
					push(@tmpz, "LINE2:temp" . $n . $LC[$n] . ":$str");
					last;
				}
			}
		} else {
			push(@tmp, "COMMENT: \\n") unless ($n + 1) % 2;
		}
	}
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=Temperatures  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Celsius",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:temp0=$HPTEMP_RRD:hptemp2_1:AVERAGE",
		"DEF:temp1=$HPTEMP_RRD:hptemp2_2:AVERAGE",
		"DEF:temp2=$HPTEMP_RRD:hptemp2_3:AVERAGE",
		"DEF:temp3=$HPTEMP_RRD:hptemp2_4:AVERAGE",
		"DEF:temp4=$HPTEMP_RRD:hptemp2_5:AVERAGE",
		"DEF:temp5=$HPTEMP_RRD:hptemp2_6:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=Temperatures  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Celsius",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:temp0=$HPTEMP_RRD:hptemp2_1:AVERAGE",
			"DEF:temp1=$HPTEMP_RRD:hptemp2_2:AVERAGE",
			"DEF:temp2=$HPTEMP_RRD:hptemp2_3:AVERAGE",
			"DEF:temp3=$HPTEMP_RRD:hptemp2_4:AVERAGE",
			"DEF:temp4=$HPTEMP_RRD:hptemp2_5:AVERAGE",
			"DEF:temp5=$HPTEMP_RRD:hptemp2_6:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /hptemp2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@tmp);
	undef(@tmpz);
	for($n = 0; $n < 6; $n++) {
		if($HPTEMP_3[$n]) {
			foreach $_ (@hplog) {
				$id = sprintf("%2d", $HPTEMP_3[$n]);
				if(/^$id  /) {
					$str = substr($_, 17, 8);
					$str = sprintf("%-8s", $str);
					push(@tmp, "LINE1:temp" . $n . $LC[$n] . ":$str");
					push(@tmp, "GPRINT:temp" . $n . ":LAST:\\: %2.0lf");
					if(!(($n + 1) % 2)) {
						push(@tmp, "COMMENT: \\n");
					} else {
						push(@tmp, "COMMENT:    ");
					}
					$str =~ s/\s+$//;
					push(@tmpz, "LINE2:temp" . $n . $LC[$n] . ":$str");
					last;
				}
			}
		} else {
			push(@tmp, "COMMENT: \\n") unless ($n + 1) % 2;
		}
	}
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=Temperatures  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Celsius",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:temp0=$HPTEMP_RRD:hptemp3_1:AVERAGE",
		"DEF:temp1=$HPTEMP_RRD:hptemp3_2:AVERAGE",
		"DEF:temp2=$HPTEMP_RRD:hptemp3_3:AVERAGE",
		"DEF:temp3=$HPTEMP_RRD:hptemp3_4:AVERAGE",
		"DEF:temp4=$HPTEMP_RRD:hptemp3_5:AVERAGE",
		"DEF:temp5=$HPTEMP_RRD:hptemp3_6:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=Temperatures  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Celsius",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:temp0=$HPTEMP_RRD:hptemp3_1:AVERAGE",
			"DEF:temp1=$HPTEMP_RRD:hptemp3_2:AVERAGE",
			"DEF:temp2=$HPTEMP_RRD:hptemp3_3:AVERAGE",
			"DEF:temp3=$HPTEMP_RRD:hptemp3_4:AVERAGE",
			"DEF:temp4=$HPTEMP_RRD:hptemp3_5:AVERAGE",
			"DEF:temp5=$HPTEMP_RRD:hptemp3_6:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /hptemp3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# LMSENS graph
# ----------------------------------------------------------------------------
sub lmsens {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @tmp;
	my @tmpz;
	my $vlabel;
	my $n;
	my $str;
	my $err;
	my @LC = (
		"#FFA500",
		"#44EEEE",
		"#44EE44",
		"#4444EE",
		"#448844",
		"#EE4444",
		"#EE44EE",
		"#EEEE44",
		"#444444",
		"#BB44EE",
		"#CCCCCC",
		"#B4B444",
		"#D3D701",
		"#E29136",
		"#DDAE8C",
		"#F29967",
	);

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG4 = $u . $myself . "4." . $when . ".png";
	my $PNG5 = $u . $myself . "5." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";
	my $PNG4z = $u . $myself . "4z." . $when . ".png";
	my $PNG5z = $u . $myself . "5z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3",
		"$PNG_DIR" . "$PNG4",
		"$PNG_DIR" . "$PNG5");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z",
			"$PNG_DIR" . "$PNG4z",
			"$PNG_DIR" . "$PNG5z");
	}

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$LMSENS_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		my $line1;
		my $line2;
		print("ERROR: while fetching $LMSENS_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		for($n = 0; $n < 2; $n++) {
			$str = "MB" . $n;
			if($SENSORS_LIST{$str}) {
				$line1 .= sprintf("%8s", $SENSORS_LIST{$str});
				$line2 .= "--------";
			}
		}
		for($n = 0; $n < 4; $n++) {
			$str = "CPU" . $n;
			if($SENSORS_LIST{$str}) {
				$line1 .= sprintf("%8s", $SENSORS_LIST{$str});
				$line2 .= "--------";
			}
		}
		for($n = 0; $n < 9; $n++) {
			$str = "FAN" . $n;
			if($SENSORS_LIST{$str}) {
				$line1 .= sprintf("%8s", $SENSORS_LIST{$str});
				$line2 .= "--------";
			}
		}
		for($n = 0; $n < 16; $n++) {
			$str = "CORE" . $n;
			if($SENSORS_LIST{$str}) {
				$line1 .= sprintf("%8s", $SENSORS_LIST{$str});
				$line2 .= "--------";
			}
		}
		for($n = 0; $n < 12; $n++) {
			$str = "VOLT" . $n;
			if($SENSORS_LIST{$str}) {
				$line1 .= sprintf("%8s", $SENSORS_LIST{$str});
				$line2 .= "--------";
			}
		}
		for($n = 0; $n < 9; $n++) {
			$str = "GPU" . $n;
			if($SENSORS_LIST{$str}) {
				$line1 .= sprintf("%8s", $SENSORS_LIST{$str});
				$line2 .= "--------";
			}
		}
		print("Time $line1\n");
		print("-----$line2\n");
		my $l;
		my $line;
		my @row;
		my $time;
		my @mb;
		my @cpu;
		my @fan;
		my @core;
		my @volt;
		my @gpu;
		for($l = 0, $time = $tb; $l < ($tb * $ts); $l++) {
			$line1 = " %2d$tc ";
			undef(@row);
			$line = @$data[$l];
			(@mb[0..2-1], @cpu[0..4-1], @fan[0..10-1], @core[0..16-1], @volt[0..10-1], @gpu[0..8-1]) = @$line;
			for($n = 0; $n < 2; $n++) {
				$str = "MB" . $n;
				if($SENSORS_LIST{$str}) {
					push(@row, $mb[$n]);
					$line1 .= "%8.1f";
				}
			}
			for($n = 0; $n < 4; $n++) {
				$str = "CPU" . $n;
				if($SENSORS_LIST{$str}) {
					push(@row, $cpu[$n]);
					$line1 .= "%8.1f";
				}
			}
			for($n = 0; $n < 9; $n++) {
				$str = "FAN" . $n;
				if($SENSORS_LIST{$str}) {
					push(@row, $fan[$n]);
					$line1 .= "%8.1f";
				}
			}
			for($n = 0; $n < 16; $n++) {
				$str = "CORE" . $n;
				if($SENSORS_LIST{$str}) {
					push(@row, $core[$n]);
					$line1 .= "%8.1f";
				}
			}
			for($n = 0; $n < 12; $n++) {
				$str = "VOLT" . $n;
				if($SENSORS_LIST{$str}) {
					push(@row, $volt[$n]);
					$line1 .= "%8.1f";
				}
			}
			for($n = 0; $n < 9; $n++) {
				$str = "GPU" . $n;
				if($SENSORS_LIST{$str}) {
					push(@row, $gpu[$n]);
					$line1 .= "%8.1f";
				}
			}
			$time = $time - (1 / $ts);
			printf("$line1 \n", $time, @row);
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		my $n2;
		graph_header($title, 2);
	}
	for($n = 0; $n < 4; $n++) {
		for($n2 = $n; $n2 < 16; $n2 += 4) {
			$str = "CORE" . $n2;
			if($SENSORS_LIST{$str}) {
				$str = sprintf("Core %2d", $n2);
				push(@tmp, "LINE1:core$n2" . $LC[$n2] . ":$str\\g");
				push(@tmp, "GPRINT:core$n2:LAST:\\:%3.0lf      ");
			}
		}
		push(@tmp, "COMMENT: \\n") unless !@tmp;
	}
	for($n = 0; $n < 16; $n++) {
		$str = "CORE" . $n;
		if($SENSORS_LIST{$str}) {
			$str = sprintf("Core %d", $n);
			push(@tmpz, "LINE2:core$n" . $LC[$n] . ":$str");
		}
	}
	# if no COREs are defined then create a blank graph
	if(!@tmp) {
		push(@tmp, "GPRINT:core0:LAST:%0.0lf");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmpz, "GPRINT:core0:LAST:%0.0lf");
	}
	if($title) {
		print("    <tr>\n");
		print("    <td valign='bottom' bgcolor='$title_bg_color'>\n");
	}
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Core temperatures  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Celsius",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:core0=$LMSENS_RRD:lmsens_core0:AVERAGE",
		"DEF:core1=$LMSENS_RRD:lmsens_core1:AVERAGE",
		"DEF:core2=$LMSENS_RRD:lmsens_core2:AVERAGE",
		"DEF:core3=$LMSENS_RRD:lmsens_core3:AVERAGE",
		"DEF:core4=$LMSENS_RRD:lmsens_core4:AVERAGE",
		"DEF:core5=$LMSENS_RRD:lmsens_core5:AVERAGE",
		"DEF:core6=$LMSENS_RRD:lmsens_core6:AVERAGE",
		"DEF:core7=$LMSENS_RRD:lmsens_core7:AVERAGE",
		"DEF:core8=$LMSENS_RRD:lmsens_core8:AVERAGE",
		"DEF:core9=$LMSENS_RRD:lmsens_core9:AVERAGE",
		"DEF:core10=$LMSENS_RRD:lmsens_core10:AVERAGE",
		"DEF:core11=$LMSENS_RRD:lmsens_core11:AVERAGE",
		"DEF:core12=$LMSENS_RRD:lmsens_core12:AVERAGE",
		"DEF:core13=$LMSENS_RRD:lmsens_core13:AVERAGE",
		"DEF:core14=$LMSENS_RRD:lmsens_core14:AVERAGE",
		"DEF:core15=$LMSENS_RRD:lmsens_core15:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Core temperatures  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Celsius",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:core0=$LMSENS_RRD:lmsens_core0:AVERAGE",
			"DEF:core1=$LMSENS_RRD:lmsens_core1:AVERAGE",
			"DEF:core2=$LMSENS_RRD:lmsens_core2:AVERAGE",
			"DEF:core3=$LMSENS_RRD:lmsens_core3:AVERAGE",
			"DEF:core4=$LMSENS_RRD:lmsens_core4:AVERAGE",
			"DEF:core5=$LMSENS_RRD:lmsens_core5:AVERAGE",
			"DEF:core6=$LMSENS_RRD:lmsens_core6:AVERAGE",
			"DEF:core7=$LMSENS_RRD:lmsens_core7:AVERAGE",
			"DEF:core8=$LMSENS_RRD:lmsens_core8:AVERAGE",
			"DEF:core9=$LMSENS_RRD:lmsens_core9:AVERAGE",
			"DEF:core10=$LMSENS_RRD:lmsens_core10:AVERAGE",
			"DEF:core11=$LMSENS_RRD:lmsens_core11:AVERAGE",
			"DEF:core12=$LMSENS_RRD:lmsens_core12:AVERAGE",
			"DEF:core13=$LMSENS_RRD:lmsens_core13:AVERAGE",
			"DEF:core14=$LMSENS_RRD:lmsens_core14:AVERAGE",
			"DEF:core15=$LMSENS_RRD:lmsens_core15:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /lmsens1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}

	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "LINE1:volt0#FFA500:Volt 0\\g");
	push(@tmp, "GPRINT:volt0:LAST:\\:%5.2lf      ");
	push(@tmp, "LINE1:volt3#4444EE:Volt 3\\g");
	push(@tmp, "GPRINT:volt3:LAST:\\:%5.2lf      ");
	push(@tmp, "LINE1:volt6#EE44EE:Volt 6\\g");
	push(@tmp, "GPRINT:volt6:LAST:\\:%5.2lf      ");
	push(@tmp, "LINE1:volt9#94C36B:Volt 9\\g");
	push(@tmp, "GPRINT:volt9:LAST:\\:%5.2lf\\n");
	push(@tmp, "LINE1:volt1#44EEEE:Volt 1\\g");
	push(@tmp, "GPRINT:volt1:LAST:\\:%5.2lf      ");
	push(@tmp, "LINE1:volt4#448844:Volt 4\\g");
	push(@tmp, "GPRINT:volt4:LAST:\\:%5.2lf      ");
	push(@tmp, "LINE1:volt7#EEEE44:Volt 7\\g");
	push(@tmp, "GPRINT:volt7:LAST:\\:%5.2lf      ");
	push(@tmp, "LINE1:volt10#3CB5B0:Volt10\\g");
	push(@tmp, "GPRINT:volt10:LAST:\\:%5.2lf\\n");
	push(@tmp, "LINE1:volt2#44EE44:Volt 2\\g");
	push(@tmp, "GPRINT:volt2:LAST:\\:%5.2lf      ");
	push(@tmp, "LINE1:volt5#EE4444:Volt 5\\g");
	push(@tmp, "GPRINT:volt5:LAST:\\:%5.2lf      ");
	push(@tmp, "LINE1:volt8#963C74:Volt 8\\g");
	push(@tmp, "GPRINT:volt8:LAST:\\:%5.2lf      ");
	push(@tmp, "LINE1:volt11#597AB7:Volt11\\g");
	push(@tmp, "GPRINT:volt11:LAST:\\:%5.2lf\\n");
	push(@tmpz, "LINE2:volt0#FFA500:Volt 0\\g");
	push(@tmpz, "LINE2:volt1#44EEEE:Volt 1\\g");
	push(@tmpz, "LINE2:volt2#44EE44:Volt 2\\g");
	push(@tmpz, "LINE2:volt3#4444EE:Volt 3\\g");
	push(@tmpz, "LINE2:volt4#448844:Volt 4\\g");
	push(@tmpz, "LINE2:volt5#EE4444:Volt 5\\g");
	push(@tmpz, "LINE2:volt6#EE44EE:Volt 6\\g");
	push(@tmpz, "LINE2:volt7#EEEE44:Volt 7\\g");
	push(@tmpz, "LINE2:volt8#963C74:Volt 8\\g");
	push(@tmpz, "LINE2:volt9#94C36B:Volt 9\\g");
	push(@tmpz, "LINE2:volt10#3CB5B0:Volt10\\g");
	push(@tmpz, "LINE2:volt11#597AB7:Volt11\\g");
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=Voltages  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Volts",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:volt0=$LMSENS_RRD:lmsens_volt0:AVERAGE",
		"DEF:volt1=$LMSENS_RRD:lmsens_volt1:AVERAGE",
		"DEF:volt2=$LMSENS_RRD:lmsens_volt2:AVERAGE",
		"DEF:volt3=$LMSENS_RRD:lmsens_volt3:AVERAGE",
		"DEF:volt4=$LMSENS_RRD:lmsens_volt4:AVERAGE",
		"DEF:volt5=$LMSENS_RRD:lmsens_volt5:AVERAGE",
		"DEF:volt6=$LMSENS_RRD:lmsens_volt6:AVERAGE",
		"DEF:volt7=$LMSENS_RRD:lmsens_volt7:AVERAGE",
		"DEF:volt8=$LMSENS_RRD:lmsens_volt8:AVERAGE",
		"DEF:volt9=$LMSENS_RRD:lmsens_volt9:AVERAGE",
		"DEF:volt10=$LMSENS_RRD:lmsens_volt10:AVERAGE",
		"DEF:volt11=$LMSENS_RRD:lmsens_volt11:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=Voltages  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Volts",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:volt0=$LMSENS_RRD:lmsens_volt0:AVERAGE",
			"DEF:volt1=$LMSENS_RRD:lmsens_volt1:AVERAGE",
			"DEF:volt2=$LMSENS_RRD:lmsens_volt2:AVERAGE",
			"DEF:volt3=$LMSENS_RRD:lmsens_volt3:AVERAGE",
			"DEF:volt4=$LMSENS_RRD:lmsens_volt4:AVERAGE",
			"DEF:volt5=$LMSENS_RRD:lmsens_volt5:AVERAGE",
			"DEF:volt6=$LMSENS_RRD:lmsens_volt6:AVERAGE",
			"DEF:volt7=$LMSENS_RRD:lmsens_volt7:AVERAGE",
			"DEF:volt8=$LMSENS_RRD:lmsens_volt8:AVERAGE",
			"DEF:volt9=$LMSENS_RRD:lmsens_volt9:AVERAGE",
			"DEF:volt10=$LMSENS_RRD:lmsens_volt10:AVERAGE",
			"DEF:volt11=$LMSENS_RRD:lmsens_volt11:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /lmsens2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "LINE1:mb0#FFA500:MB 0\\g");
	push(@tmp, "GPRINT:mb0:LAST:\\:%3.0lf   ");
	push(@tmp, "LINE1:cpu0#4444EE:CPU 0\\g");
	push(@tmp, "GPRINT:cpu0:LAST:\\:%3.0lf   ");
	push(@tmp, "LINE1:cpu2#EE44EE:CPU 2\\g");
	push(@tmp, "GPRINT:cpu2:LAST:\\:%3.0lf\\n");
	push(@tmp, "LINE1:mb1#44EEEE:MB 1\\g");
	push(@tmp, "GPRINT:mb1:LAST:\\:%3.0lf   ");
	push(@tmp, "LINE1:cpu1#EEEE44:CPU 1\\g");
	push(@tmp, "GPRINT:cpu1:LAST:\\:%3.0lf   ");
	push(@tmp, "LINE1:cpu3#44EE44:CPU 3\\g");
	push(@tmp, "GPRINT:cpu3:LAST:\\:%3.0lf\\n");
	push(@tmpz, "LINE2:mb0#FFA500:MB 0");
	push(@tmpz, "LINE2:mb1#44EEEE:MB 1");
	push(@tmpz, "LINE2:cpu0#4444EE:CPU 0");
	push(@tmpz, "LINE2:cpu1#EEEE44:CPU 1");
	push(@tmpz, "LINE2:cpu2#EE44EE:CPU 2");
	push(@tmpz, "LINE2:cpu3#44EE44:CPU 3");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=MB & CPU temperatures  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Celsius",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:mb0=$LMSENS_RRD:lmsens_mb0:AVERAGE",
		"DEF:mb1=$LMSENS_RRD:lmsens_mb1:AVERAGE",
		"DEF:cpu0=$LMSENS_RRD:lmsens_cpu0:AVERAGE",
		"DEF:cpu1=$LMSENS_RRD:lmsens_cpu1:AVERAGE",
		"DEF:cpu2=$LMSENS_RRD:lmsens_cpu2:AVERAGE",
		"DEF:cpu3=$LMSENS_RRD:lmsens_cpu3:AVERAGE",
		"COMMENT: \\n",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=MB & CPU temperatures  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Celsius",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:mb0=$LMSENS_RRD:lmsens_mb0:AVERAGE",
			"DEF:mb1=$LMSENS_RRD:lmsens_mb1:AVERAGE",
			"DEF:cpu0=$LMSENS_RRD:lmsens_cpu0:AVERAGE",
			"DEF:cpu1=$LMSENS_RRD:lmsens_cpu1:AVERAGE",
			"DEF:cpu2=$LMSENS_RRD:lmsens_cpu2:AVERAGE",
			"DEF:cpu3=$LMSENS_RRD:lmsens_cpu3:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /lmsens3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "LINE1:fan0#FFA500:Fan 0\\g");
	push(@tmp, "GPRINT:fan0:LAST:\\:%5.0lf");
	push(@tmp, "LINE1:fan3#4444EE:Fan 3\\g");
	push(@tmp, "GPRINT:fan3:LAST:\\:%5.0lf");
	push(@tmp, "LINE1:fan6#EE44EE:Fan 6\\g");
	push(@tmp, "GPRINT:fan6:LAST:\\:%5.0lf\\n");
	push(@tmp, "LINE1:fan1#44EEEE:Fan 1\\g");
	push(@tmp, "GPRINT:fan1:LAST:\\:%5.0lf");
	push(@tmp, "LINE1:fan4#448844:Fan 4\\g");
	push(@tmp, "GPRINT:fan4:LAST:\\:%5.0lf");
	push(@tmp, "LINE1:fan7#EEEE44:Fan 7\\g");
	push(@tmp, "GPRINT:fan7:LAST:\\:%5.0lf\\n");
	push(@tmp, "LINE1:fan2#44EE44:Fan 2\\g");
	push(@tmp, "GPRINT:fan2:LAST:\\:%5.0lf");
	push(@tmp, "LINE1:fan5#EE4444:Fan 5\\g");
	push(@tmp, "GPRINT:fan5:LAST:\\:%5.0lf");
	push(@tmp, "LINE1:fan8#963C74:Fan 8\\g");
	push(@tmp, "GPRINT:fan8:LAST:\\:%5.0lf\\n");
	push(@tmpz, "LINE2:fan0#FFA500:Fan 0");
	push(@tmpz, "LINE2:fan1#44EEEE:Fan 1");
	push(@tmpz, "LINE2:fan2#44EE44:Fan 2");
	push(@tmpz, "LINE2:fan3#4444EE:Fan 3");
	push(@tmpz, "LINE2:fan4#448844:Fan 4");
	push(@tmpz, "LINE2:fan5#EE4444:Fan 5");
	push(@tmpz, "LINE2:fan6#EE44EE:Fan 6");
	push(@tmpz, "LINE2:fan7#EEEE44:Fan 7");
	push(@tmpz, "LINE2:fan8#963C74:Fan 8");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG4",
		"--title=Fan speeds  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=RPM",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:fan0=$LMSENS_RRD:lmsens_fan0:AVERAGE",
		"DEF:fan1=$LMSENS_RRD:lmsens_fan1:AVERAGE",
		"DEF:fan2=$LMSENS_RRD:lmsens_fan2:AVERAGE",
		"DEF:fan3=$LMSENS_RRD:lmsens_fan3:AVERAGE",
		"DEF:fan4=$LMSENS_RRD:lmsens_fan4:AVERAGE",
		"DEF:fan5=$LMSENS_RRD:lmsens_fan5:AVERAGE",
		"DEF:fan6=$LMSENS_RRD:lmsens_fan6:AVERAGE",
		"DEF:fan7=$LMSENS_RRD:lmsens_fan7:AVERAGE",
		"DEF:fan8=$LMSENS_RRD:lmsens_fan8:AVERAGE",
		"COMMENT: \\n",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG4: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG4z",
			"--title=Fan speeds  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=RPM",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:fan0=$LMSENS_RRD:lmsens_fan0:AVERAGE",
			"DEF:fan1=$LMSENS_RRD:lmsens_fan1:AVERAGE",
			"DEF:fan2=$LMSENS_RRD:lmsens_fan2:AVERAGE",
			"DEF:fan3=$LMSENS_RRD:lmsens_fan3:AVERAGE",
			"DEF:fan4=$LMSENS_RRD:lmsens_fan4:AVERAGE",
			"DEF:fan5=$LMSENS_RRD:lmsens_fan5:AVERAGE",
			"DEF:fan6=$LMSENS_RRD:lmsens_fan6:AVERAGE",
			"DEF:fan7=$LMSENS_RRD:lmsens_fan7:AVERAGE",
			"DEF:fan8=$LMSENS_RRD:lmsens_fan8:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG4z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /lmsens4/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG4z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG4 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG4 . "'>\n");
		}
	}

	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "LINE1:gpu0#FFA500:GPU 0\\g");
	push(@tmp, "GPRINT:gpu0:LAST:\\:%3.0lf  ");
	push(@tmp, "LINE1:gpu3#4444EE:GPU 3\\g");
	push(@tmp, "GPRINT:gpu3:LAST:\\:%3.0lf  ");
	push(@tmp, "LINE1:gpu6#EE44EE:GPU 6\\g");
	push(@tmp, "GPRINT:gpu6:LAST:\\:%3.0lf\\n");
	push(@tmp, "LINE1:gpu1#44EEEE:GPU 1\\g");
	push(@tmp, "GPRINT:gpu1:LAST:\\:%3.0lf  ");
	push(@tmp, "LINE1:gpu4#448844:GPU 4\\g");
	push(@tmp, "GPRINT:gpu4:LAST:\\:%3.0lf  ");
	push(@tmp, "LINE1:gpu7#EEEE44:GPU 7\\g");
	push(@tmp, "GPRINT:gpu7:LAST:\\:%3.0lf\\n");
	push(@tmp, "LINE1:gpu2#44EE44:GPU 2\\g");
	push(@tmp, "GPRINT:gpu2:LAST:\\:%3.0lf  ");
	push(@tmp, "LINE1:gpu5#EE4444:GPU 5\\g");
	push(@tmp, "GPRINT:gpu5:LAST:\\:%3.0lf  ");
	push(@tmp, "LINE1:gpu8#963C74:GPU 8\\g");
	push(@tmp, "GPRINT:gpu8:LAST:\\:%3.0lf\\n");
	push(@tmpz, "LINE2:gpu0#FFA500:GPU 0\\g");
	push(@tmpz, "LINE2:gpu1#44EEEE:GPU 1\\g");
	push(@tmpz, "LINE2:gpu2#44EE44:GPU 2\\g");
	push(@tmpz, "LINE2:gpu3#4444EE:GPU 3\\g");
	push(@tmpz, "LINE2:gpu4#448844:GPU 4\\g");
	push(@tmpz, "LINE2:gpu5#EE4444:GPU 5\\g");
	push(@tmpz, "LINE2:gpu6#EE44EE:GPU 6\\g");
	push(@tmpz, "LINE2:gpu7#EEEE44:GPU 7\\g");
	push(@tmpz, "LINE2:gpu8#963C74:GPU 8\\g");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG5",
		"--title=GPU temperatures  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Celsius",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:gpu0=$LMSENS_RRD:lmsens_gpu0:AVERAGE",
		"DEF:gpu1=$LMSENS_RRD:lmsens_gpu1:AVERAGE",
		"DEF:gpu2=$LMSENS_RRD:lmsens_gpu2:AVERAGE",
		"DEF:gpu3=$LMSENS_RRD:lmsens_gpu3:AVERAGE",
		"DEF:gpu4=$LMSENS_RRD:lmsens_gpu4:AVERAGE",
		"DEF:gpu5=$LMSENS_RRD:lmsens_gpu5:AVERAGE",
		"DEF:gpu6=$LMSENS_RRD:lmsens_gpu6:AVERAGE",
		"DEF:gpu7=$LMSENS_RRD:lmsens_gpu7:AVERAGE",
		"DEF:gpu8=$LMSENS_RRD:lmsens_gpu8:AVERAGE",
		"COMMENT: \\n",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG5: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG5z",
			"--title=GPU temperatures  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Celsius",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:gpu0=$LMSENS_RRD:lmsens_gpu0:AVERAGE",
			"DEF:gpu1=$LMSENS_RRD:lmsens_gpu1:AVERAGE",
			"DEF:gpu2=$LMSENS_RRD:lmsens_gpu2:AVERAGE",
			"DEF:gpu3=$LMSENS_RRD:lmsens_gpu3:AVERAGE",
			"DEF:gpu4=$LMSENS_RRD:lmsens_gpu4:AVERAGE",
			"DEF:gpu5=$LMSENS_RRD:lmsens_gpu5:AVERAGE",
			"DEF:gpu6=$LMSENS_RRD:lmsens_gpu6:AVERAGE",
			"DEF:gpu7=$LMSENS_RRD:lmsens_gpu7:AVERAGE",
			"DEF:gpu8=$LMSENS_RRD:lmsens_gpu8:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG5z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /lmsens5/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG5z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG5 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG5 . "'>\n");
		}
	}


	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# NVIDIA graph
# ----------------------------------------------------------------------------
sub nvidia {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @tmp;
	my @tmpz;
	my $n;
	my $err;
	my @LC = (
		"#FFA500",
		"#44EEEE",
		"#44EE44",
		"#4444EE",
		"#448844",
		"#EE4444",
		"#EE44EE",
		"#EEEE44",
		"#963C74",
	);

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$NVIDIA_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		my $line2;
		my $line3;
		print("ERROR: while fetching $NVIDIA_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		print("    ");
		for($n = 0; $n < $NVIDIA_MAX; $n++) {
			print("   NVIDIA card $n ");
		}
		print("\n");
		for($n = 0; $n < $NVIDIA_MAX; $n++) {
			$line2 .= "   Temp  GPU  Mem";
			$line3 .= "-----------------";
		}
		print("Time$line2\n");
		print("----$line3 \n");
		my $line;
		my @row;
		my $time;
		my $n2;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			$time = $time - (1 / $ts);
			printf(" %2d$tc ", $time);
			undef($line1);
			undef(@row);
			for($n2 = 0; $n2 < $NVIDIA_MAX; $n2++) {
				push(@row, @$line[$n2]);
				push(@row, @$line[$n2 + 9]);
				push(@row, @$line[$n2 + 18]);
				$line1 .= "   %3d %3d%% %3d%%";
			}
			print(sprintf($line1, @row));
			print("\n");
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}
	for($n = 0; $n < 9; $n++) {
		if($n < $NVIDIA_MAX) {
			push(@tmp, "LINE1:temp" . $n . $LC[$n] . ":Card $n");
			push(@tmpz, "LINE2:temp" . $n . $LC[$n] . ":Card $n");
			push(@tmp, "GPRINT:temp" . $n . ":LAST:             Current\\: %2.0lf");
			push(@tmp, "GPRINT:temp" . $n . ":AVERAGE:   Average\\: %2.0lf");
			push(@tmp, "GPRINT:temp" . $n . ":MIN:   Min\\: %2.0lf");
			push(@tmp, "GPRINT:temp" . $n . ":MAX:   Max\\: %2.0lf\\n");
		} else {
			push(@tmp, "COMMENT: \\n");
		}
	}

	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Temperatures  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Celsius",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:temp0=$NVIDIA_RRD:nvidia_temp0:AVERAGE",
		"DEF:temp1=$NVIDIA_RRD:nvidia_temp1:AVERAGE",
		"DEF:temp2=$NVIDIA_RRD:nvidia_temp2:AVERAGE",
		"DEF:temp3=$NVIDIA_RRD:nvidia_temp3:AVERAGE",
		"DEF:temp4=$NVIDIA_RRD:nvidia_temp4:AVERAGE",
		"DEF:temp5=$NVIDIA_RRD:nvidia_temp5:AVERAGE",
		"DEF:temp6=$NVIDIA_RRD:nvidia_temp6:AVERAGE",
		"DEF:temp7=$NVIDIA_RRD:nvidia_temp7:AVERAGE",
		"DEF:temp8=$NVIDIA_RRD:nvidia_temp8:AVERAGE",
		@tmp,
		"COMMENT: \\n",
		"COMMENT: \\n");
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Temperatures  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Celsius",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:temp0=$NVIDIA_RRD:nvidia_temp0:AVERAGE",
			"DEF:temp1=$NVIDIA_RRD:nvidia_temp1:AVERAGE",
			"DEF:temp2=$NVIDIA_RRD:nvidia_temp2:AVERAGE",
			"DEF:temp3=$NVIDIA_RRD:nvidia_temp3:AVERAGE",
			"DEF:temp4=$NVIDIA_RRD:nvidia_temp4:AVERAGE",
			"DEF:temp5=$NVIDIA_RRD:nvidia_temp5:AVERAGE",
			"DEF:temp6=$NVIDIA_RRD:nvidia_temp6:AVERAGE",
			"DEF:temp7=$NVIDIA_RRD:nvidia_temp7:AVERAGE",
			"DEF:temp8=$NVIDIA_RRD:nvidia_temp8:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /nvidia1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "LINE1:gpu0#FFA500:Card 0\\g");
	push(@tmp, "GPRINT:gpu0:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:gpu3#4444EE:Card 3\\g");
	push(@tmp, "GPRINT:gpu3:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:gpu6#EE44EE:Card 6\\g");
	push(@tmp, "GPRINT:gpu6:LAST:\\:%3.0lf%%\\n");
	push(@tmp, "LINE1:gpu1#44EEEE:Card 1\\g");
	push(@tmp, "GPRINT:gpu1:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:gpu4#448844:Card 4\\g");
	push(@tmp, "GPRINT:gpu4:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:gpu7#EEEE44:Card 7\\g");
	push(@tmp, "GPRINT:gpu7:LAST:\\:%3.0lf%%\\n");
	push(@tmp, "LINE1:gpu2#44EE44:Card 2\\g");
	push(@tmp, "GPRINT:gpu2:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:gpu5#EE4444:Card 5\\g");
	push(@tmp, "GPRINT:gpu5:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:gpu8#963C74:Card 8\\g");
	push(@tmp, "GPRINT:gpu8:LAST:\\:%3.0lf%%\\n");
	push(@tmpz, "LINE2:gpu0#FFA500:Card 0");
	push(@tmpz, "LINE2:gpu3#4444EE:Card 3");
	push(@tmpz, "LINE2:gpu6#EE44EE:Card 6");
	push(@tmpz, "LINE2:gpu1#44EEEE:Card 1");
	push(@tmpz, "LINE2:gpu4#448844:Card 4");
	push(@tmpz, "LINE2:gpu7#EEEE44:Card 7");
	push(@tmpz, "LINE2:gpu2#44EE44:Card 2");
	push(@tmpz, "LINE2:gpu5#EE4444:Card 5");
	push(@tmpz, "LINE2:gpu8#963C74:Card 8");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=GPU usage  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Percent",
		"--width=$width",
		"--height=$height",
		"--upper-limit=100",
		"--lower-limit=0",
		"--rigid",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:gpu0=$NVIDIA_RRD:nvidia_gpu0:AVERAGE",
		"DEF:gpu1=$NVIDIA_RRD:nvidia_gpu1:AVERAGE",
		"DEF:gpu2=$NVIDIA_RRD:nvidia_gpu2:AVERAGE",
		"DEF:gpu3=$NVIDIA_RRD:nvidia_gpu3:AVERAGE",
		"DEF:gpu4=$NVIDIA_RRD:nvidia_gpu4:AVERAGE",
		"DEF:gpu5=$NVIDIA_RRD:nvidia_gpu5:AVERAGE",
		"DEF:gpu6=$NVIDIA_RRD:nvidia_gpu6:AVERAGE",
		"DEF:gpu7=$NVIDIA_RRD:nvidia_gpu7:AVERAGE",
		"DEF:gpu8=$NVIDIA_RRD:nvidia_gpu8:AVERAGE",
		"COMMENT: \\n",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=GPU usage  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Percent",
			"--width=$width",
			"--height=$height",
			"--upper-limit=100",
			"--lower-limit=0",
			"--rigid",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:gpu0=$NVIDIA_RRD:nvidia_gpu0:AVERAGE",
			"DEF:gpu1=$NVIDIA_RRD:nvidia_gpu1:AVERAGE",
			"DEF:gpu2=$NVIDIA_RRD:nvidia_gpu2:AVERAGE",
			"DEF:gpu3=$NVIDIA_RRD:nvidia_gpu3:AVERAGE",
			"DEF:gpu4=$NVIDIA_RRD:nvidia_gpu4:AVERAGE",
			"DEF:gpu5=$NVIDIA_RRD:nvidia_gpu5:AVERAGE",
			"DEF:gpu6=$NVIDIA_RRD:nvidia_gpu6:AVERAGE",
			"DEF:gpu7=$NVIDIA_RRD:nvidia_gpu7:AVERAGE",
			"DEF:gpu8=$NVIDIA_RRD:nvidia_gpu8:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /nvidia2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "LINE1:mem0#FFA500:Card 0\\g");
	push(@tmp, "GPRINT:mem0:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:mem3#4444EE:Card 3\\g");
	push(@tmp, "GPRINT:mem3:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:mem6#EE44EE:Card 6\\g");
	push(@tmp, "GPRINT:mem6:LAST:\\:%3.0lf%%\\n");
	push(@tmp, "LINE1:mem1#44EEEE:Card 1\\g");
	push(@tmp, "GPRINT:mem1:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:mem4#448844:Card 4\\g");
	push(@tmp, "GPRINT:mem4:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:mem7#EEEE44:Card 7\\g");
	push(@tmp, "GPRINT:mem7:LAST:\\:%3.0lf%%\\n");
	push(@tmp, "LINE1:mem2#44EE44:Card 2\\g");
	push(@tmp, "GPRINT:mem2:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:mem5#EE4444:Card 5\\g");
	push(@tmp, "GPRINT:mem5:LAST:\\:%3.0lf%%");
	push(@tmp, "LINE1:mem8#963C74:Card 8\\g");
	push(@tmp, "GPRINT:mem8:LAST:\\:%3.0lf%%\\n");
	push(@tmpz, "LINE2:mem0#FFA500:Card 0");
	push(@tmpz, "LINE2:mem3#4444EE:Card 3");
	push(@tmpz, "LINE2:mem6#EE44EE:Card 6");
	push(@tmpz, "LINE2:mem1#44EEEE:Card 1");
	push(@tmpz, "LINE2:mem4#448844:Card 4");
	push(@tmpz, "LINE2:mem7#EEEE44:Card 7");
	push(@tmpz, "LINE2:mem2#44EE44:Card 2");
	push(@tmpz, "LINE2:mem5#EE4444:Card 5");
	push(@tmpz, "LINE2:mem8#963C74:Card 8");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=Memory usage  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Percent",
		"--width=$width",
		"--height=$height",
		"--upper-limit=100",
		"--lower-limit=0",
		"--rigid",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:mem0=$NVIDIA_RRD:nvidia_mem0:AVERAGE",
		"DEF:mem1=$NVIDIA_RRD:nvidia_mem1:AVERAGE",
		"DEF:mem2=$NVIDIA_RRD:nvidia_mem2:AVERAGE",
		"DEF:mem3=$NVIDIA_RRD:nvidia_mem3:AVERAGE",
		"DEF:mem4=$NVIDIA_RRD:nvidia_mem4:AVERAGE",
		"DEF:mem5=$NVIDIA_RRD:nvidia_mem5:AVERAGE",
		"DEF:mem6=$NVIDIA_RRD:nvidia_mem6:AVERAGE",
		"DEF:mem7=$NVIDIA_RRD:nvidia_mem7:AVERAGE",
		"DEF:mem8=$NVIDIA_RRD:nvidia_mem8:AVERAGE",
		"COMMENT: \\n",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=Memory usage  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Percent",
			"--width=$width",
			"--height=$height",
			"--upper-limit=100",
			"--lower-limit=0",
			"--rigid",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:mem0=$NVIDIA_RRD:nvidia_mem0:AVERAGE",
			"DEF:mem1=$NVIDIA_RRD:nvidia_mem1:AVERAGE",
			"DEF:mem2=$NVIDIA_RRD:nvidia_mem2:AVERAGE",
			"DEF:mem3=$NVIDIA_RRD:nvidia_mem3:AVERAGE",
			"DEF:mem4=$NVIDIA_RRD:nvidia_mem4:AVERAGE",
			"DEF:mem5=$NVIDIA_RRD:nvidia_mem5:AVERAGE",
			"DEF:mem6=$NVIDIA_RRD:nvidia_mem6:AVERAGE",
			"DEF:mem7=$NVIDIA_RRD:nvidia_mem7:AVERAGE",
			"DEF:mem8=$NVIDIA_RRD:nvidia_mem8:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /nvidia3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# DISK graph
# ----------------------------------------------------------------------------
sub disk {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @tmp;
	my @tmpz;
	my $n;
	my $str;
	my $err;
	my @LC = (
		"#FFA500",
		"#44EEEE",
		"#44EE44",
		"#4444EE",
		"#448844",
		"#EE4444",
		"#EE44EE",
		"#EEEE44",
		"#444444",
		"#BB44EE",
		"#CCCCCC",
		"#B4B444",
		"#D3D701",
		"#E29136",
		"#DDAE8C",
		"#F29967",
	);

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$DISK_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		my $line1;
		my $line2;
		my $line3;
		print("ERROR: while fetching $DISK_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		for($n = 0; $n < scalar(@DISK_LIST); $n++) {
			$str = sprintf(" DISK %d               ", $n + 1);
			$line1 .= $str;
			$str = sprintf(" Temp Realloc Pending ", $n + 1);
			$line2 .= $str;
			$line3 .=      "---------------------";
		}

		print("     $line1\n");
		print("Time $line2\n");
		print("-------$line3\n");
		my $line;
		my @row;
		my $time;
		my $n2;
		my $from;
		my $to;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			$time = $time - (1 / $ts);
			printf(" %2d$tc ", $time);
			for($n2 = 0; $n2 < scalar(@DISK_LIST); $n2++) {
				$from = $n2 * scalar(@DISK_LIST);
				$to = $from + scalar(@DISK_LIST);
				my ($temp, $realloc, $pending) = @$line[$from..$to];
				@row = ($temp, $realloc, $pending);
				printf(" %4.0f %7.0f %7.0f ", @row);
			}
			print("\n");
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}
	for($n = 0; $n < 8; $n++) {
		if($DISK_LIST[$n]) {
			$str = sprintf("%-20s", $DISK_LIST[$n]);
			push(@tmp, "LINE1:hd" . $n . $LC[$n] . ":$str");
			push(@tmpz, "LINE2:hd" . $n . $LC[$n] . ":$DISK_LIST[$n]");
			push(@tmp, "GPRINT:hd" . $n . ":LAST:Current\\: %2.0lf");
			push(@tmp, "GPRINT:hd" . $n . ":AVERAGE:   Average\\: %2.0lf");
			push(@tmp, "GPRINT:hd" . $n . ":MIN:   Min\\: %2.0lf");
			push(@tmp, "GPRINT:hd" . $n . ":MAX:   Max\\: %2.0lf\\n");
		} else {
			push(@tmp, "COMMENT: \\n");
		}
	}
	for($n = 3; $n < 8; $n++) {
		if($DISK_LIST[$n]) {
			push(@tmp, "COMMENT: \\n");
			push(@tmp, "COMMENT: \\n");
		}
	}

	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Disk drives temperatures  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Celsius",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:hd0=$DISK_RRD:disk_hd0_temp:AVERAGE",
		"DEF:hd1=$DISK_RRD:disk_hd1_temp:AVERAGE",
		"DEF:hd2=$DISK_RRD:disk_hd2_temp:AVERAGE",
		"DEF:hd3=$DISK_RRD:disk_hd3_temp:AVERAGE",
		"DEF:hd4=$DISK_RRD:disk_hd4_temp:AVERAGE",
		"DEF:hd5=$DISK_RRD:disk_hd5_temp:AVERAGE",
		"DEF:hd6=$DISK_RRD:disk_hd6_temp:AVERAGE",
		"DEF:hd7=$DISK_RRD:disk_hd7_temp:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Disk drives temperatures  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Celsius",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:hd0=$DISK_RRD:disk_hd0_temp:AVERAGE",
			"DEF:hd1=$DISK_RRD:disk_hd1_temp:AVERAGE",
			"DEF:hd2=$DISK_RRD:disk_hd2_temp:AVERAGE",
			"DEF:hd3=$DISK_RRD:disk_hd3_temp:AVERAGE",
			"DEF:hd4=$DISK_RRD:disk_hd4_temp:AVERAGE",
			"DEF:hd5=$DISK_RRD:disk_hd5_temp:AVERAGE",
			"DEF:hd6=$DISK_RRD:disk_hd6_temp:AVERAGE",
			"DEF:hd7=$DISK_RRD:disk_hd7_temp:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /disk1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@tmp);
	undef(@tmpz);
	for($n = 0; $n < 8; $n++) {
		if($DISK_LIST[$n]) {
			$str = sprintf("%-20s", $DISK_LIST[$n]);
			push(@tmp, "LINE1:rsc" . $n . $LC[$n] . ":$str\\g");
			push(@tmpz, "LINE2:rsc" . $n . $LC[$n] . ":$DISK_LIST[$n]");
			push(@tmp, "GPRINT:rsc" . $n . ":LAST:          Current\\: %8.0lf\\n");
		} else {
			if($n < 3) {
				push(@tmp, "COMMENT: \\n");
			}
		}
	}
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=Reallocated Sector Count  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Sectors",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:rsc0=$DISK_RRD:disk_hd0_smart1:AVERAGE",
		"DEF:rsc1=$DISK_RRD:disk_hd1_smart1:AVERAGE",
		"DEF:rsc2=$DISK_RRD:disk_hd2_smart1:AVERAGE",
		"DEF:rsc3=$DISK_RRD:disk_hd3_smart1:AVERAGE",
		"DEF:rsc4=$DISK_RRD:disk_hd4_smart1:AVERAGE",
		"DEF:rsc5=$DISK_RRD:disk_hd5_smart1:AVERAGE",
		"DEF:rsc6=$DISK_RRD:disk_hd6_smart1:AVERAGE",
		"DEF:rsc7=$DISK_RRD:disk_hd7_smart1:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=Reallocated Sector Count  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Sectors",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:rsc0=$DISK_RRD:disk_hd0_smart1:AVERAGE",
			"DEF:rsc1=$DISK_RRD:disk_hd1_smart1:AVERAGE",
			"DEF:rsc2=$DISK_RRD:disk_hd2_smart1:AVERAGE",
			"DEF:rsc3=$DISK_RRD:disk_hd3_smart1:AVERAGE",
			"DEF:rsc4=$DISK_RRD:disk_hd4_smart1:AVERAGE",
			"DEF:rsc5=$DISK_RRD:disk_hd5_smart1:AVERAGE",
			"DEF:rsc6=$DISK_RRD:disk_hd6_smart1:AVERAGE",
			"DEF:rsc7=$DISK_RRD:disk_hd7_smart1:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /disk2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@tmp);
	undef(@tmpz);
	for($n = 0; $n < 8; $n++) {
		if($DISK_LIST[$n]) {
			$str = sprintf("%-20s", $DISK_LIST[$n]);
			push(@tmp, "LINE1:cps" . $n . $LC[$n] . ":$str\\g");
			push(@tmpz, "LINE2:cps" . $n . $LC[$n] . ":$DISK_LIST[$n]");
			push(@tmp, "GPRINT:cps" . $n . ":LAST:          Current\\: %8.0lf\\n");
		} else {
			if($n < 3) {
				push(@tmp, "COMMENT: \\n");
			}
		}
	}
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=Current Pending Sector  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Sectors",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:cps0=$DISK_RRD:disk_hd0_smart2:AVERAGE",
		"DEF:cps1=$DISK_RRD:disk_hd1_smart2:AVERAGE",
		"DEF:cps2=$DISK_RRD:disk_hd2_smart2:AVERAGE",
		"DEF:cps3=$DISK_RRD:disk_hd3_smart2:AVERAGE",
		"DEF:cps4=$DISK_RRD:disk_hd4_smart2:AVERAGE",
		"DEF:cps5=$DISK_RRD:disk_hd5_smart2:AVERAGE",
		"DEF:cps6=$DISK_RRD:disk_hd6_smart2:AVERAGE",
		"DEF:cps7=$DISK_RRD:disk_hd7_smart2:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=Current Pending Sector  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Sectors",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:cps0=$DISK_RRD:disk_hd0_smart2:AVERAGE",
			"DEF:cps1=$DISK_RRD:disk_hd1_smart2:AVERAGE",
			"DEF:cps2=$DISK_RRD:disk_hd2_smart2:AVERAGE",
			"DEF:cps3=$DISK_RRD:disk_hd3_smart2:AVERAGE",
			"DEF:cps4=$DISK_RRD:disk_hd4_smart2:AVERAGE",
			"DEF:cps5=$DISK_RRD:disk_hd5_smart2:AVERAGE",
			"DEF:cps6=$DISK_RRD:disk_hd6_smart2:AVERAGE",
			"DEF:cps7=$DISK_RRD:disk_hd7_smart2:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /disk3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# FS graph
# ----------------------------------------------------------------------------
sub fs {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my $graph_title;
	my $vlabel;
	my @tmp;
	my @tmpz;
	my @area;
	my @areaz;
	my @line;
	my @linez;
	my @riglim;
	my $n;
	my $str;
	my $err;
	my @AC = (
		"#444444",
		"#44EEEE",
		"#EE44EE",
		"#EEEE44",
		"#44EE44",
		"#4444EE",
		"#448844",
		"#B4B444",
		"#FFA500",
		"#BB44EE",
	);
	my @LC = (
		"#444444",
		"#00EEEE",
		"#EE00EE",
		"#EEEE00",
		"#00EE00",
		"#0000EE",
		"#448844",
		"#B4B444",
		"#FFA500",
		"#BB44EE",
	);

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$FS_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		my $n2;
		my $line1;
		my $line2;
		print("ERROR: while fetching $FS_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		$line1 = "      /   Swap";
		$line2 = "--------------";
		for($n = 0; $n < scalar(@FS_LIST); $n++) {
			$str = $FS_DESC[$n];
			$str = $FS_LIST[$n] unless $str;
			$line1 .= "    " . $str;
			$line2 .= "----";
			for($n2 = 0; $n2 < length($str); $n2++) {
				$line2 .= "-";
			}
		}

		$line1 .= "   I/O Reads  I/O Writes  Sect.Reads  Sect.Writes";
		$line2 .= "-------------------------------------------------";
		print("Time $line1 \n");
		print("-----$line2\n");
		my $line;
		my @row;
		my $time;
		my $l;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			$time = $time - (1 / $ts);
			my ($root, $swap) = @$line;
			printf(" %2d$tc   %4.1f%%   %4.1f%%", $time, $root, $swap);
			for($n2 = 0; $n2 < scalar(@FS_LIST); $n2++) {
				my $used = @$line[$n2 + 2];
				$str = $FS_DESC[$n2];
				$str = $FS_LIST[$n2] unless $str;
				$l = length($str) - 1;
				$l = ($l, 4)[$l < 4];	# min()
				printf "   %$l.1f%%", $used;
			}
			my ($ior, $iow, $scr, $scw) = @$line[11..14];
			printf "  %10d  %10d  %10d   %10d", $ior, $iow, $scr, $scw;
			print("\n");
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}
	$str = sprintf("%-23s", "/");
	push(@area, "AREA:root#EE4444:$str");
	push(@area, "GPRINT:root" . ":LAST:Cur\\: %4.1lf%%");
	push(@area, "GPRINT:root" . ":AVERAGE:   Avg\\: %4.1lf%%");
	push(@area, "GPRINT:root" . ":MIN:   Min\\: %4.1lf%%");
	push(@area, "GPRINT:root" . ":MAX:   Max\\: %4.1lf%%\\n");
	$str = sprintf("%-23s", "swap");
	push(@area, "LINE2:swap#CCCCCC:$str");
	push(@area, "GPRINT:swap" . ":LAST:Cur\\: %4.1lf%%");
	push(@area, "GPRINT:swap" . ":AVERAGE:   Avg\\: %4.1lf%%");
	push(@area, "GPRINT:swap" . ":MIN:   Min\\: %4.1lf%%");
	push(@area, "GPRINT:swap" . ":MAX:   Max\\: %4.1lf%%\\n");
	push(@line, "LINE2:root#EE0000:");
	push(@line, "LINE2:swap#CCCCCC:");
	push(@areaz, "AREA:root#EE4444:/");
	push(@areaz, "LINE2:root#CCCCCC:swap");
	for($n = 0; $n < 9; $n++) {
		if($FS_LIST[$n]) {
			$str = $FS_DESC[$n];
			$str = $FS_LIST[$n] unless $str;
			push(@areaz, "AREA:mnt" . ($n + 1) . $AC[$n] . ":$str");
			$str = sprintf("%-23s", $str);
			push(@area, "AREA:mnt" . ($n + 1) . $AC[$n] . ":$str");
			push(@area, "GPRINT:mnt" . ($n + 1) . ":LAST:Cur\\: %4.1lf%%");
			push(@area, "GPRINT:mnt" . ($n + 1) . ":AVERAGE:   Avg\\: %4.1lf%%");
			push(@area, "GPRINT:mnt" . ($n + 1) . ":MIN:   Min\\: %4.1lf%%");
			push(@area, "GPRINT:mnt" . ($n + 1) . ":MAX:   Max\\: %4.1lf%%\\n");
		} else {
			push(@area, "COMMENT: \\n");
		}
	}
	push(@linez, "LINE2:root#EE0000:");
	push(@linez, "LINE2:root#CCCCCC:");
	for($n = 0; $n < 9; $n++) {
		if($FS_LIST[$n]) {
			push(@line, "LINE2:mnt" . ($n + 1) . $LC[$n] . ":");
			push(@linez, "LINE2:mnt" . ($n + 1) . $LC[$n] . ":");
		}
	}

	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@area = @areaz;
		@line = @linez;
		push(@line, "COMMENT: \\n");
		push(@line, "COMMENT: \\n");
		push(@line, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Filesystems usage  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Percent (%)",
		"--width=$width",
		"--height=$height",
		"--upper-limit=100",
		"--lower-limit=0",
		"--rigid",
		@VERSION12,
		@graph_colors,
		"DEF:root=$FS_RRD:fs_root:AVERAGE",
		"DEF:swap=$FS_RRD:fs_swap:AVERAGE",
		"DEF:mnt1=$FS_RRD:fs_mnt1:AVERAGE",
		"DEF:mnt2=$FS_RRD:fs_mnt2:AVERAGE",
		"DEF:mnt3=$FS_RRD:fs_mnt3:AVERAGE",
		"DEF:mnt4=$FS_RRD:fs_mnt4:AVERAGE",
		"DEF:mnt5=$FS_RRD:fs_mnt5:AVERAGE",
		"DEF:mnt6=$FS_RRD:fs_mnt6:AVERAGE",
		"DEF:mnt7=$FS_RRD:fs_mnt7:AVERAGE",
		"DEF:mnt8=$FS_RRD:fs_mnt8:AVERAGE",
		"DEF:mnt9=$FS_RRD:fs_mnt9:AVERAGE",
		@area,
		"AREA:root#EE4444:",
		"LINE2:swap#CCCCCC:",
		@line);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Filesystems usage  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Percent (%)",
			"--width=$width",
			"--height=$height",
			"--upper-limit=100",
			"--lower-limit=0",
			"--rigid",
			@VERSION12,
			@graph_colors,
			"DEF:root=$FS_RRD:fs_root:AVERAGE",
			"DEF:swap=$FS_RRD:fs_swap:AVERAGE",
			"DEF:mnt1=$FS_RRD:fs_mnt1:AVERAGE",
			"DEF:mnt2=$FS_RRD:fs_mnt2:AVERAGE",
			"DEF:mnt3=$FS_RRD:fs_mnt3:AVERAGE",
			"DEF:mnt4=$FS_RRD:fs_mnt4:AVERAGE",
			"DEF:mnt5=$FS_RRD:fs_mnt5:AVERAGE",
			"DEF:mnt6=$FS_RRD:fs_mnt6:AVERAGE",
			"DEF:mnt7=$FS_RRD:fs_mnt7:AVERAGE",
			"DEF:mnt8=$FS_RRD:fs_mnt8:AVERAGE",
			"DEF:mnt9=$FS_RRD:fs_mnt9:AVERAGE",
			@areaz,
			"AREA:root#EE4444:",
			"LINE2:swap#CCCCCC:",
			@linez);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /fs1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	if($FS2_RIGID eq 1) {
		push(@riglim, "--upper-limit=$FS2_LIMIT");
	} else {
		if($FS2_RIGID eq 2) {
			push(@riglim, "--upper-limit=$FS2_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:r_cnt#44EE44:Reads");
	push(@tmp, "GPRINT:r_cnt:LAST:           Current\\: %8.0lf\\n");
	push(@tmp, "AREA:w_cnt#4444EE:Writes");
	push(@tmp, "GPRINT:w_cnt:LAST:          Current\\: %8.0lf\\n");
	push(@tmp, "AREA:w_cnt#4444EE:");
	push(@tmp, "AREA:r_cnt#44EE44:");
	push(@tmp, "LINE1:w_cnt#0000EE");
	push(@tmp, "LINE1:r_cnt#00EE00");
	push(@tmpz, "AREA:r_cnt#44EE44:Reads");
	push(@tmpz, "AREA:w_cnt#4444EE:Writes");
	push(@tmpz, "AREA:w_cnt#4444EE:");
	push(@tmpz, "AREA:r_cnt#44EE44:");
	push(@tmpz, "LINE1:w_cnt#0000EE");
	push(@tmpz, "LINE1:r_cnt#00EE00");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=I/O activity  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=I/O /sec",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:r_cnt=$FS_RRD:fs_read_cnt:AVERAGE",
		"DEF:w_cnt=$FS_RRD:fs_write_cnt:AVERAGE",
		"COMMENT: \\n",
		@tmp,
		"COMMENT: \\n");
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=I/O activiy  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=I/O /sec",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:r_cnt=$FS_RRD:fs_read_cnt:AVERAGE",
			"DEF:w_cnt=$FS_RRD:fs_write_cnt:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /fs2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@riglim);
	if($FS3_RIGID eq 1) {
		push(@riglim, "--upper-limit=$FS3_LIMIT");
	} else {
		if($FS3_RIGID eq 2) {
			push(@riglim, "--upper-limit=$FS3_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	undef(@tmp);
	undef(@tmpz);
	if($os eq "Linux") {
	   	$graph_title = "Sector activity  (1$when)";
		$vlabel = "Sectors/sec";
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "AREA:r_sec#44EE44:Read");
		push(@tmp, "GPRINT:r_sec:LAST:            Current\\: %8.0lf\\n");
		push(@tmp, "AREA:w_sec#4444EE:Written");
		push(@tmp, "GPRINT:w_sec:LAST:         Current\\: %8.0lf\\n");
		push(@tmp, "AREA:w_sec#4444EE:");
		push(@tmp, "AREA:r_sec#44EE44:");
		push(@tmp, "LINE1:w_sec#0000EE");
		push(@tmp, "LINE1:r_sec#00EE00");
		push(@tmp, "COMMENT: \\n");
		push(@tmpz, "AREA:r_sec#44EE44:Read");
		push(@tmpz, "AREA:w_sec#4444EE:Written");
		push(@tmpz, "AREA:w_sec#4444EE:");
		push(@tmpz, "AREA:r_sec#44EE44:");
		push(@tmpz, "LINE1:w_sec#0000EE");
		push(@tmpz, "LINE1:r_sec#00EE00");
	} elsif($os eq "FreeBSD") {
	   	$graph_title = "Data activity  (1$when)";
		$vlabel = "KB/sec";
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "AREA:r_sec#44EE44:KB Read");
		push(@tmp, "GPRINT:r_sec:LAST:          Current\\: %8.0lf\\n");
		push(@tmp, "AREA:w_sec#4444EE:KB Written");
		push(@tmp, "GPRINT:w_sec:LAST:       Current\\: %8.0lf\\n");
		push(@tmp, "AREA:w_sec#4444EE:");
		push(@tmp, "AREA:r_sec#44EE44:");
		push(@tmp, "LINE1:w_sec#0000EE");
		push(@tmp, "LINE1:r_sec#00EE00");
		push(@tmp, "COMMENT: \\n");
		push(@tmpz, "AREA:r_sec#44EE44:KB Read");
		push(@tmpz, "AREA:w_sec#4444EE:KB Written");
		push(@tmpz, "AREA:w_sec#4444EE:");
		push(@tmpz, "AREA:r_sec#44EE44:");
		push(@tmpz, "LINE1:w_sec#0000EE");
		push(@tmpz, "LINE1:r_sec#00EE00");
	}
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=$graph_title",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=$vlabel",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:r_sec=$FS_RRD:fs_read_sec:AVERAGE",
		"DEF:w_sec=$FS_RRD:fs_write_sec:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=$graph_title",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=$vlabel",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:r_sec=$FS_RRD:fs_read_sec:AVERAGE",
			"DEF:w_sec=$FS_RRD:fs_write_sec:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /fs3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# NET graph
# ----------------------------------------------------------------------------
sub net {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @riglim;
	my $PNG1;
	my $PNG2;
	my $PNG3;
	my $PNG1z;
	my $PNG2z;
	my $PNG3z;
	my $netname;
	my @tmp;
	my @tmpz;
	my $n;
	my $str;
	my $err;

	$title = !$silent ? $title : "";

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$NET_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		print("ERROR: while fetching $NET_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		print("       ");
		for($n = 0; $n < scalar(@NET_LIST); $n++) {
			print("$NET_LIST[$n] ($NET_DESC[$n])                          ");
		}
		print("\nTime");
		for($n = 0; $n < scalar(@NET_LIST); $n++) {
			print("   KB/s_I  KB/s_O  Pk/s_I  Pk/s_O  Er/s_I  Er/s_O");
		}
		print(" \n----");
		for($n = 0; $n < scalar(@NET_LIST); $n++) {
			print("-------------------------------------------------");
		}
		print " \n";
		my $line;
		my @row;
		my $time;
		my $n2;
		my $from;
		my $to;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			$time = $time - (1 / $ts);
			printf(" %2d$tc", $time);
			for($n2 = 0; $n2 < scalar(@NET_LIST); $n2++) {
				$from = $n2 * 6;
				$to = $from + 6;
				my ($ki, $ko, $pi, $po, $ei, $eo) = @$line[$from..$to];
				$ki /= 1024;
				$ko /= 1024;
				$pi /= 1024;
				$po /= 1024;
				$ei /= 1024;
				$eo /= 1024;
				@row = ($ki, $ko, $pi, $po, $ei, $eo);
				printf "   %6d  %6d  %6d  %6d  %6d  %6d", @row;
			}
			print(" \n");
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	for($n = 0; $n < scalar(@NET_LIST); $n++) {
		$PNG1 = $u . $myself . $n . "1." . $when . ".png";
		$PNG2 = $u . $myself . $n . "2." . $when . ".png";
		$PNG3 = $u . $myself . $n . "3." . $when . ".png";
		unlink("$PNG_DIR" . $PNG1);
		unlink("$PNG_DIR" . $PNG2);
		unlink("$PNG_DIR" . $PNG3);
		if($ENABLE_ZOOM eq "Y") {
			$PNG1z = $u . $myself . $n . "1z." . $when . ".png";
			$PNG2z = $u . $myself . $n . "2z." . $when . ".png";
			$PNG3z = $u . $myself . $n . "3z." . $when . ".png";
			unlink("$PNG_DIR" . $PNG1z);
			unlink("$PNG_DIR" . $PNG2z);
			unlink("$PNG_DIR" . $PNG3z);
		}

		if($title) {
			if($n) {
			print("    <br>\n");
			}
			graph_header($NET_LIST[$n] . " " . $title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}

		undef(@riglim);
		if($NET_RIGID[$n] eq 1) {
			push(@riglim, "--upper-limit=$NET_LIMIT[$n]");
		} else {
			if($NET_RIGID[$n] eq 2) {
				push(@riglim, "--upper-limit=$NET_LIMIT[$n]");
				push(@riglim, "--rigid");
			}
		}
		undef(@tmp);
		undef(@tmpz);
		push(@tmp, "AREA:B_in#44EE44:KB/s Input");
		push(@tmp, "GPRINT:K_in:LAST:     Current\\: %5.0lf");
		push(@tmp, "GPRINT:K_in:AVERAGE: Average\\: %5.0lf");
		push(@tmp, "GPRINT:K_in:MIN:    Min\\: %5.0lf");
		push(@tmp, "GPRINT:K_in:MAX:    Max\\: %5.0lf\\n");
		push(@tmp, "AREA:B_out#4444EE:KB/s Output");
		push(@tmp, "GPRINT:K_out:LAST:    Current\\: %5.0lf");
		push(@tmp, "GPRINT:K_out:AVERAGE: Average\\: %5.0lf");
		push(@tmp, "GPRINT:K_out:MIN:    Min\\: %5.0lf");
		push(@tmp, "GPRINT:K_out:MAX:    Max\\: %5.0lf\\n");
		push(@tmp, "AREA:B_out#4444EE:");
		push(@tmp, "AREA:B_in#44EE44:");
		push(@tmp, "LINE1:B_out#0000EE");
		push(@tmp, "LINE1:B_in#00EE00");
		push(@tmpz, "AREA:B_in#44EE44:Input");
		push(@tmpz, "AREA:B_out#4444EE:Output");
		push(@tmpz, "AREA:B_out#4444EE:");
		push(@tmpz, "AREA:B_in#44EE44:");
		push(@tmpz, "LINE1:B_out#0000EE");
		push(@tmpz, "LINE1:B_in#00EE00");
		($width, $height) = split('x', $GRAPH_SIZE{main});
		if($silent =~ /imagetag/) {
			($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
			($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
			@tmp = @tmpz;
			push(@tmp, "COMMENT: \\n");
		}
		RRDs::graph("$PNG_DIR" . "$PNG1",
			"--title=$NET_LIST[$n] $NET_DESC[$n] traffic  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=bytes/sec",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:B_in=$NET_RRD:net" . $n . "_bytes_in:AVERAGE",
			"DEF:B_out=$NET_RRD:net" . $n . "_bytes_out:AVERAGE",
			"CDEF:K_in=B_in,1024,/",
			"CDEF:K_out=B_out,1024,/",
			"COMMENT: \\n",
			@tmp,
			"COMMENT: \\n",
			"COMMENT: \\n",
			);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
		if($ENABLE_ZOOM eq "Y") {
			($width, $height) = split('x', $GRAPH_SIZE{zoom});
			RRDs::graph("$PNG_DIR" . "$PNG1z",
				"--title=$NET_LIST[$n] $NET_TYPE[$n] traffic  (1$when)",
				"--start=-1$when",
				"--imgformat=PNG",
				"--vertical-label=bytes/sec",
				"--width=$width",
				"--height=$height",
				@riglim,
				"--lower-limit=0",
				@VERSION12,
				@graph_colors,
				"DEF:B_in=$NET_RRD:net" . $n . "_bytes_in:AVERAGE",
				"DEF:B_out=$NET_RRD:net" . $n . "_bytes_out:AVERAGE",
				@tmpz);
			$err = RRDs::error;
			print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
		}
		$netname="net" . $n . "1";
		if($title || ($silent =~ /imagetag/ && $graph =~ /$netname/)) {
			if($ENABLE_ZOOM eq "Y") {
				print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
			} else {
				print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
			}
		}
		if($title) {
			print("    </td>\n");
			print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
		}

		undef(@tmp);
		undef(@tmpz);
		push(@tmp, "AREA:p_in#44EE44:Input");
		push(@tmp, "AREA:p_out#4444EE:Output");
		push(@tmp, "AREA:p_out#4444EE:");
		push(@tmp, "AREA:p_in#44EE44:");
		push(@tmp, "LINE1:p_out#0000EE");
		push(@tmp, "LINE1:p_in#00EE00");
		push(@tmpz, "AREA:p_in#44EE44:Input");
		push(@tmpz, "AREA:p_out#4444EE:Output");
		push(@tmpz, "AREA:p_out#4444EE:");
		push(@tmpz, "AREA:p_in#44EE44:");
		push(@tmpz, "LINE1:p_out#0000EE");
		push(@tmpz, "LINE1:p_in#00EE00");
		($width, $height) = split('x', $GRAPH_SIZE{small});
		if($silent =~ /imagetag/) {
			($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
			($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
			@tmp = @tmpz;
			push(@tmp, "COMMENT: \\n");
			push(@tmp, "COMMENT: \\n");
			push(@tmp, "COMMENT: \\n");
		}
		RRDs::graph("$PNG_DIR" . "$PNG2",
			"--title=$NET_LIST[$n] packet traffic  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Packets/sec",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:p_in=$NET_RRD:net" . $n . "_packs_in:AVERAGE",
			"DEF:p_out=$NET_RRD:net" . $n . "_packs_out:AVERAGE",
			@tmp);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
		if($ENABLE_ZOOM eq "Y") {
			($width, $height) = split('x', $GRAPH_SIZE{zoom});
			RRDs::graph("$PNG_DIR" . "$PNG2z",
				"--title=$NET_LIST[$n] packet traffic  (1$when)",
				"--start=-1$when",
				"--imgformat=PNG",
				"--vertical-label=Packets/sec",
				"--width=$width",
				"--height=$height",
				"--lower-limit=0",
				@VERSION12,
				@VERSION12_small,
				@graph_colors,
				"DEF:p_in=$NET_RRD:net" . $n . "_packs_in:AVERAGE",
				"DEF:p_out=$NET_RRD:net" . $n . "_packs_out:AVERAGE",
				@tmpz);
			$err = RRDs::error;
			print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
		}
		$netname="net" . $n . "2";
		if($title || ($silent =~ /imagetag/ && $graph =~ /$netname/)) {
			if($ENABLE_ZOOM eq "Y") {
				print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
			} else {
				print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
			}
		}

		undef(@tmp);
		undef(@tmpz);
		push(@tmp, "AREA:e_in#44EE44:Input");
		push(@tmp, "AREA:e_out#4444EE:Output");
		push(@tmp, "AREA:e_out#4444EE:");
		push(@tmp, "AREA:e_in#44EE44:");
		push(@tmp, "LINE1:e_out#0000EE");
		push(@tmp, "LINE1:e_in#00EE00");
		push(@tmpz, "AREA:e_in#44EE44:Input");
		push(@tmpz, "AREA:e_out#4444EE:Output");
		push(@tmpz, "AREA:e_out#4444EE:");
		push(@tmpz, "AREA:e_in#44EE44:");
		push(@tmpz, "LINE1:e_out#0000EE");
		push(@tmpz, "LINE1:e_in#00EE00");
		($width, $height) = split('x', $GRAPH_SIZE{small});
		if($silent =~ /imagetag/) {
			($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
			($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
			@tmp = @tmpz;
			push(@tmp, "COMMENT: \\n");
			push(@tmp, "COMMENT: \\n");
			push(@tmp, "COMMENT: \\n");
		}
		RRDs::graph("$PNG_DIR" . "$PNG3",
			"--title=$NET_LIST[$n] traffic error  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Errors/sec",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:e_in=$NET_RRD:net" . $n . "_error_in:AVERAGE",
			"DEF:e_out=$NET_RRD:net" . $n . "_error_out:AVERAGE",
			@tmp);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
		if($ENABLE_ZOOM eq "Y") {
			($width, $height) = split('x', $GRAPH_SIZE{zoom});
			RRDs::graph("$PNG_DIR" . "$PNG3z",
				"--title=$NET_LIST[$n] traffic error  (1$when)",
				"--start=-1$when",
				"--imgformat=PNG",
				"--vertical-label=Errors/sec",
				"--width=$width",
				"--height=$height",
				"--lower-limit=0",
				@VERSION12,
				@VERSION12_small,
				@graph_colors,
				"DEF:e_in=$NET_RRD:net" . $n . "_error_in:AVERAGE",
				"DEF:e_out=$NET_RRD:net" . $n . "_error_out:AVERAGE",
				@tmpz);
			$err = RRDs::error;
			print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
		}
		$netname="net" . $n . "3";
		if($title || ($silent =~ /imagetag/ && $graph =~ /$netname/)) {
			if($ENABLE_ZOOM eq "Y") {
				print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
			} else {
				print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
			}
		}

		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
	}
	return 1;
}

# SERV graph
# ----------------------------------------------------------------------------
sub serv {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my $vlabel;
	my @tmp;
	my @tmpz;
	my $n;
	my $str;
	my $err;

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$SERV_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		print("ERROR: while fetching $SERV_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		if($SERV_MODE eq "I") {
			print "Values expressed as incremental or cumulative hits.\n";
		}
		print("Time    SSH     FTP  Telnet    IMAP   Samba     Fax    CUPS    POP3    SMTP    Spam   Virus\n");
		print("------------------------------------------------------------------------------------------- \n");
		my $line;
		my @row;
		my $time;
		my $from = 0;
		my $to;
		if($SERV_MODE eq "L") {
			$from = 15;
		}
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			$time = $time - (1 / $ts);
			$to = $from + 10;
			my ($ssh, $ftp, $telnet, $imap, $smb, $fax, $cups, $pop3, $smtp, $spam, $virus) = @$line[$from..$to];
			@row = ($ssh, $ftp, $telnet, $imap, $smb, $fax, $cups, $pop3, $smtp, $spam, $virus);
			if($SERV_MODE eq "I") {
				printf(" %2d$tc %6d  %6d  %6d  %6d  %6d  %6d  %6d  %6d  %6d  %6d  %6d\n", $time, @row);
			} elsif($SERV_MODE eq "L") {
				printf(" %2d$tc %6.2f  %6.2f  %6.2f  %6.2f  %6.2f  %6.2f  %6.2f  %6.2f  %6.2f  %6.2f  %6.2f\n", $time, @row);
			}
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}
	if($SERV_MODE eq "L") {
		$vlabel = "Accesses/sec";
		push(@tmp, "AREA:l_ssh#4444EE:SSH");
		push(@tmp, "GPRINT:l_ssh:LAST:        Current\\: %3.2lf");
		push(@tmp, "GPRINT:l_ssh:AVERAGE:   Average\\: %3.2lf");
		push(@tmp, "GPRINT:l_ssh:MIN:   Min\\: %3.2lf");
		push(@tmp, "GPRINT:l_ssh:MAX:   Max\\: %3.2lf\\n");
		push(@tmp, "AREA:l_ftp#44EE44:FTP");
		push(@tmp, "GPRINT:l_ftp:LAST:        Current\\: %3.2lf");
		push(@tmp, "GPRINT:l_ftp:AVERAGE:   Average\\: %3.2lf");
		push(@tmp, "GPRINT:l_ftp:MIN:   Min\\: %3.2lf");
		push(@tmp, "GPRINT:l_ftp:MAX:   Max\\: %3.2lf\\n");
		push(@tmp, "AREA:l_telnet#EE44EE:Telnet");
		push(@tmp, "GPRINT:l_telnet:LAST:     Current\\: %3.2lf");
		push(@tmp, "GPRINT:l_telnet:AVERAGE:   Average\\: %3.2lf");
		push(@tmp, "GPRINT:l_telnet:MIN:   Min\\: %3.2lf");
		push(@tmp, "GPRINT:l_telnet:MAX:   Max\\: %3.2lf\\n");
		push(@tmp, "AREA:l_imap#44EEEE:IMAP");
		push(@tmp, "GPRINT:l_imap:LAST:       Current\\: %3.2lf");
		push(@tmp, "GPRINT:l_imap:AVERAGE:   Average\\: %3.2lf");
		push(@tmp, "GPRINT:l_imap:MIN:   Min\\: %3.2lf");
		push(@tmp, "GPRINT:l_imap:MAX:   Max\\: %3.2lf\\n");
		push(@tmp, "AREA:l_smb#EEEE44:Samba");
		push(@tmp, "GPRINT:l_smb:LAST:      Current\\: %3.2lf");
		push(@tmp, "GPRINT:l_smb:AVERAGE:   Average\\: %3.2lf");
		push(@tmp, "GPRINT:l_smb:MIN:   Min\\: %3.2lf");
		push(@tmp, "GPRINT:l_smb:MAX:   Max\\: %3.2lf\\n");
		push(@tmp, "AREA:l_fax#FFA500:Fax");
		push(@tmp, "GPRINT:l_fax:LAST:        Current\\: %3.2lf");
		push(@tmp, "GPRINT:l_fax:AVERAGE:   Average\\: %3.2lf");
		push(@tmp, "GPRINT:l_fax:MIN:   Min\\: %3.2lf");
		push(@tmp, "GPRINT:l_fax:MAX:   Max\\: %3.2lf\\n");
		push(@tmp, "AREA:l_cups#444444:CUPS");
		push(@tmp, "GPRINT:l_cups:LAST:       Current\\: %3.2lf");
		push(@tmp, "GPRINT:l_cups:AVERAGE:   Average\\: %3.2lf");
		push(@tmp, "GPRINT:l_cups:MIN:   Min\\: %3.2lf");
		push(@tmp, "GPRINT:l_cups:MAX:   Max\\: %3.2lf\\n");
		push(@tmp, "LINE2:l_ssh#4444EE");
		push(@tmp, "LINE2:l_ftp#44EE44");
		push(@tmp, "LINE2:l_telnet#EE44EE");
		push(@tmp, "LINE2:l_imap#44EEEE");
		push(@tmp, "LINE2:l_smb#EEEE44");
		push(@tmp, "LINE2:l_fax#FFA500");
		push(@tmp, "LINE2:l_cups#444444");

		push(@tmpz, "AREA:l_ssh#4444EE:SSH");
		push(@tmpz, "AREA:l_ftp#44EE44:FTP");
		push(@tmpz, "AREA:l_telnet#EE44EE:Telnet");
		push(@tmpz, "AREA:l_imap#44EEEE:IMAP");
		push(@tmpz, "AREA:l_smb#EEEE44:Samba");
		push(@tmpz, "AREA:l_fax#FFA500:Fax");
		push(@tmpz, "AREA:l_cups#444444:CUPS");
		push(@tmpz, "LINE2:l_ssh#4444EE");
		push(@tmpz, "LINE2:l_ftp#44EE44");
		push(@tmpz, "LINE2:l_telnet#EE44EE");
		push(@tmpz, "LINE2:l_imap#44EEEE");
		push(@tmpz, "LINE2:l_smb#EEEE44");
		push(@tmpz, "LINE2:l_fax#FFA500");
		push(@tmpz, "LINE2:l_cups#444444");
	} else {
		$vlabel = "Incremental hits";
		push(@tmp, "AREA:i_ssh#4444EE:SSH");
		push(@tmp, "GPRINT:i_ssh:LAST:        Current\\: %5.0lf");
		push(@tmp, "GPRINT:i_ssh:AVERAGE:   Average\\: %5.0lf");
		push(@tmp, "GPRINT:i_ssh:MIN:   Min\\: %5.0lf");
		push(@tmp, "GPRINT:i_ssh:MAX:   Max\\: %5.0lf\\n");
		push(@tmp, "AREA:i_ftp#44EE44:FTP");
		push(@tmp, "GPRINT:i_ftp:LAST:        Current\\: %5.0lf");
		push(@tmp, "GPRINT:i_ftp:AVERAGE:   Average\\: %5.0lf");
		push(@tmp, "GPRINT:i_ftp:MIN:   Min\\: %5.0lf");
		push(@tmp, "GPRINT:i_ftp:MAX:   Max\\: %5.0lf\\n");
		push(@tmp, "AREA:i_telnet#EE44EE:Telnet");
		push(@tmp, "GPRINT:i_telnet:LAST:     Current\\: %5.0lf");
		push(@tmp, "GPRINT:i_telnet:AVERAGE:   Average\\: %5.0lf");
		push(@tmp, "GPRINT:i_telnet:MIN:   Min\\: %5.0lf");
		push(@tmp, "GPRINT:i_telnet:MAX:   Max\\: %5.0lf\\n");
		push(@tmp, "AREA:i_imap#44EEEE:IMAP");
		push(@tmp, "GPRINT:i_imap:LAST:       Current\\: %5.0lf");
		push(@tmp, "GPRINT:i_imap:AVERAGE:   Average\\: %5.0lf");
		push(@tmp, "GPRINT:i_imap:MIN:   Min\\: %5.0lf");
		push(@tmp, "GPRINT:i_imap:MAX:   Max\\: %5.0lf\\n");
		push(@tmp, "AREA:i_smb#EEEE44:Samba");
		push(@tmp, "GPRINT:i_smb:LAST:      Current\\: %5.0lf");
		push(@tmp, "GPRINT:i_smb:AVERAGE:   Average\\: %5.0lf");
		push(@tmp, "GPRINT:i_smb:MIN:   Min\\: %5.0lf");
		push(@tmp, "GPRINT:i_smb:MAX:   Max\\: %5.0lf\\n");
		push(@tmp, "AREA:i_fax#FFA500:Fax");
		push(@tmp, "GPRINT:i_fax:LAST:        Current\\: %5.0lf");
		push(@tmp, "GPRINT:i_fax:AVERAGE:   Average\\: %5.0lf");
		push(@tmp, "GPRINT:i_fax:MIN:   Min\\: %5.0lf");
		push(@tmp, "GPRINT:i_fax:MAX:   Max\\: %5.0lf\\n");
		push(@tmp, "AREA:i_cups#444444:CUPS");
		push(@tmp, "GPRINT:i_cups:LAST:       Current\\: %5.0lf");
		push(@tmp, "GPRINT:i_cups:AVERAGE:   Average\\: %5.0lf");
		push(@tmp, "GPRINT:i_cups:MIN:   Min\\: %5.0lf");
		push(@tmp, "GPRINT:i_cups:MAX:   Max\\: %5.0lf\\n");
		push(@tmp, "LINE2:i_ssh#4444EE");
		push(@tmp, "LINE2:i_ftp#44EE44");
		push(@tmp, "LINE2:i_telnet#EE44EE");
		push(@tmp, "LINE2:i_imap#44EEEE");
		push(@tmp, "LINE2:i_smb#EEEE44");
		push(@tmp, "LINE2:i_fax#FFA500");
		push(@tmp, "LINE2:i_cups#444444");

		push(@tmpz, "AREA:i_ssh#4444EE:SSH");
		push(@tmpz, "AREA:i_ftp#44EE44:FTP");
		push(@tmpz, "AREA:i_telnet#EE44EE:Telnet");
		push(@tmpz, "AREA:i_imap#44EEEE:IMAP");
		push(@tmpz, "AREA:i_smb#EEEE44:Samba");
		push(@tmpz, "AREA:i_fax#FFA500:Fax");
		push(@tmpz, "AREA:i_cups#444444:CUPS");
		push(@tmpz, "LINE2:i_ssh#4444EE");
		push(@tmpz, "LINE2:i_ftp#44EE44");
		push(@tmpz, "LINE2:i_telnet#EE44EE");
		push(@tmpz, "LINE2:i_imap#44EEEE");
		push(@tmpz, "LINE2:i_smb#EEEE44");
		push(@tmpz, "LINE2:i_fax#FFA500");
		push(@tmpz, "LINE2:i_cups#444444");
	}

	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Services demand  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=$vlabel",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:i_ssh=$SERV_RRD:serv_i_ssh:AVERAGE",
		"DEF:i_ftp=$SERV_RRD:serv_i_ftp:AVERAGE",
		"DEF:i_telnet=$SERV_RRD:serv_i_telnet:AVERAGE",
		"DEF:i_imap=$SERV_RRD:serv_i_imap:AVERAGE",
		"DEF:i_smb=$SERV_RRD:serv_i_smb:AVERAGE",
		"DEF:i_fax=$SERV_RRD:serv_i_fax:AVERAGE",
		"DEF:i_cups=$SERV_RRD:serv_i_cups:AVERAGE",
		"DEF:l_ssh=$SERV_RRD:serv_l_ssh:AVERAGE",
		"DEF:l_ftp=$SERV_RRD:serv_l_ftp:AVERAGE",
		"DEF:l_telnet=$SERV_RRD:serv_l_telnet:AVERAGE",
		"DEF:l_imap=$SERV_RRD:serv_l_imap:AVERAGE",
		"DEF:l_smb=$SERV_RRD:serv_l_smb:AVERAGE",
		"DEF:l_fax=$SERV_RRD:serv_l_fax:AVERAGE",
		"DEF:l_cups=$SERV_RRD:serv_l_cups:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Services demand  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=$vlabel",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:i_ssh=$SERV_RRD:serv_i_ssh:AVERAGE",
			"DEF:i_ftp=$SERV_RRD:serv_i_ftp:AVERAGE",
			"DEF:i_telnet=$SERV_RRD:serv_i_telnet:AVERAGE",
			"DEF:i_imap=$SERV_RRD:serv_i_imap:AVERAGE",
			"DEF:i_smb=$SERV_RRD:serv_i_smb:AVERAGE",
			"DEF:i_fax=$SERV_RRD:serv_i_fax:AVERAGE",
			"DEF:i_cups=$SERV_RRD:serv_i_cups:AVERAGE",
			"DEF:l_ssh=$SERV_RRD:serv_l_ssh:AVERAGE",
			"DEF:l_ftp=$SERV_RRD:serv_l_ftp:AVERAGE",
			"DEF:l_telnet=$SERV_RRD:serv_l_telnet:AVERAGE",
			"DEF:l_imap=$SERV_RRD:serv_l_imap:AVERAGE",
			"DEF:l_smb=$SERV_RRD:serv_l_smb:AVERAGE",
			"DEF:l_fax=$SERV_RRD:serv_l_fax:AVERAGE",
			"DEF:l_cups=$SERV_RRD:serv_l_cups:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /serv1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@tmp);
	undef(@tmpz);
	if($SERV_MODE eq "L") {
		$vlabel = "Accesses/sec";
		push(@tmp, "AREA:l_pop3#44EE44:POP3");
		push(@tmp, "GPRINT:l_pop3:LAST:            Current\\: %3.2lf\\n");
		push(@tmpz, "AREA:l_pop3#44EE44:POP3");
	} else {
		$vlabel = "Incremental hits";
		push(@tmp, "AREA:i_pop3#44EE44:POP3");
		push(@tmp, "GPRINT:i_pop3:LAST:            Current\\: %5.0lf\\n");
		push(@tmpz, "AREA:i_pop3#44EE44:POP3");
	}
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=POP3 service  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=$vlabel",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:i_pop3=$SERV_RRD:serv_i_pop3:AVERAGE",
		"DEF:l_pop3=$SERV_RRD:serv_l_pop3:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=POP3 service  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=$vlabel",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:i_pop3=$SERV_RRD:serv_i_pop3:AVERAGE",
			"DEF:l_pop3=$SERV_RRD:serv_l_pop3:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /serv2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@tmp);
	undef(@tmpz);
	if($SERV_MODE eq "L") {
		$vlabel = "Accesses/sec";
		push(@tmp, "AREA:l_smtp#44EEEE:SMTP");
		push(@tmp, "GPRINT:l_smtp:LAST:              Current\\: %3.2lf\\n");
		push(@tmp, "AREA:l_spam#EEEE44:Spam");
		push(@tmp, "GPRINT:l_spam:LAST:              Current\\: %3.2lf\\n");
		push(@tmp, "AREA:l_virus#EE4444:Virus");
		push(@tmp, "GPRINT:l_virus:LAST:             Current\\: %3.2lf\\n");
		push(@tmp, "LINE2:l_smtp#44EEEE");
		push(@tmp, "LINE2:l_spam#EEEE44");
		push(@tmp, "LINE2:l_virus#EE4444");

		push(@tmpz, "AREA:l_smtp#44EEEE:SMTP");
		push(@tmpz, "AREA:l_spam#EEEE44:Spam");
		push(@tmpz, "AREA:l_virus#EE4444:Virus");
		push(@tmpz, "LINE2:l_smtp#44EEEE");
		push(@tmpz, "LINE2:l_spam#EEEE44");
		push(@tmpz, "LINE2:l_virus#EE4444");
	} else {
		$vlabel = "Incremental hits";
		push(@tmp, "AREA:i_smtp#44EEEE:SMTP");
		push(@tmp, "GPRINT:i_smtp:LAST:            Current\\: %5.0lf\\n");
		push(@tmp, "AREA:i_spam#EEEE44:Spam");
		push(@tmp, "GPRINT:i_spam:LAST:            Current\\: %5.0lf\\n");
		push(@tmp, "AREA:i_virus#EE4444:Virus");
		push(@tmp, "GPRINT:i_virus:LAST:           Current\\: %5.0lf\\n");
		push(@tmp, "LINE2:i_smtp#44EEEE");
		push(@tmp, "LINE2:i_spam#EEEE44");
		push(@tmp, "LINE2:i_virus#EE4444");

		push(@tmpz, "AREA:i_smtp#44EEEE:SMTP");
		push(@tmpz, "AREA:i_spam#EEEE44:Spam");
		push(@tmpz, "AREA:i_virus#EE4444:Virus");
		push(@tmpz, "LINE2:i_smtp#44EEEE");
		push(@tmpz, "LINE2:i_spam#EEEE44");
		push(@tmpz, "LINE2:i_virus#EE4444");
	}
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=Mail service  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=$vlabel",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:i_smtp=$SERV_RRD:serv_i_smtp:AVERAGE",
		"DEF:i_spam=$SERV_RRD:serv_i_spam:AVERAGE",
		"DEF:i_virus=$SERV_RRD:serv_i_virus:AVERAGE",
		"DEF:l_smtp=$SERV_RRD:serv_l_smtp:AVERAGE",
		"DEF:l_spam=$SERV_RRD:serv_l_spam:AVERAGE",
		"DEF:l_virus=$SERV_RRD:serv_l_virus:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		undef(@tmp);
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=Mail service  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=$vlabel",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:i_smtp=$SERV_RRD:serv_i_smtp:AVERAGE",
			"DEF:i_spam=$SERV_RRD:serv_i_spam:AVERAGE",
			"DEF:i_virus=$SERV_RRD:serv_i_virus:AVERAGE",
			"DEF:l_smtp=$SERV_RRD:serv_l_smtp:AVERAGE",
			"DEF:l_spam=$SERV_RRD:serv_l_spam:AVERAGE",
			"DEF:l_virus=$SERV_RRD:serv_l_virus:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /serv3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# MAIL graph
# ----------------------------------------------------------------------------
sub mail {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @riglim;
	my @tmp;
	my @tmpz;
	my $n;
	my $str;
	my $err;

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$MAIL_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		print("ERROR: while fetching $MAIL_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		print("Time  In.Conn Out.Conn  Receivd   Delivd  Bytes.R  Bytes.D  Rejectd     Spam    Virus   Queued\n");
		print("---------------------------------------------------------------------------------------------- \n");
		my $line;
		my @row;
		my $time;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			$time = $time - (1 / $ts);
			my ($in, $out, $recvd, $delvd, $bytes_recvd, $bytes_delvd, $rejtd, $spam, $virus, undef, $queued) = @$line;
			@row = ($in, $out, $recvd, $delvd, $bytes_recvd, $bytes_delvd, $rejtd, $spam, $virus, $queued);
			printf(" %2d$tc  %7.2f  %7.2f  %7.2f  %7.2f  %7.2f  %7.2f  %7.2f  %7.2f  %7.2f  %7.2f\n", $time, @row);
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}
	push(@tmp, "AREA:in#44EE44:In Connections");
	push(@tmp, "GPRINT:in:LAST:   Current\\: %3.2lf");
	push(@tmp, "GPRINT:in:AVERAGE:  Average\\: %3.2lf");
	push(@tmp, "GPRINT:in:MIN:  Min\\: %3.2lf");
	push(@tmp, "GPRINT:in:MAX:  Max\\: %3.2lf\\n");
	push(@tmp, "AREA:rejtd#EE4444:Rejected");
	push(@tmp, "GPRINT:rejtd:LAST:         Current\\: %3.2lf");
	push(@tmp, "GPRINT:rejtd:AVERAGE:  Average\\: %3.2lf");
	push(@tmp, "GPRINT:rejtd:MIN:  Min\\: %3.2lf");
	push(@tmp, "GPRINT:rejtd:MAX:  Max\\: %3.2lf\\n");
	push(@tmp, "AREA:recvd#448844:Received");
	push(@tmp, "GPRINT:recvd:LAST:         Current\\: %3.2lf");
	push(@tmp, "GPRINT:recvd:AVERAGE:  Average\\: %3.2lf");
	push(@tmp, "GPRINT:recvd:MIN:  Min\\: %3.2lf");
	push(@tmp, "GPRINT:recvd:MAX:  Max\\: %3.2lf\\n");
	push(@tmp, "AREA:spam#EEEE44:Spam");
	push(@tmp, "GPRINT:spam:LAST:             Current\\: %3.2lf");
	push(@tmp, "GPRINT:spam:AVERAGE:  Average\\: %3.2lf");
	push(@tmp, "GPRINT:spam:MIN:  Min\\: %3.2lf");
	push(@tmp, "GPRINT:spam:MAX:  Max\\: %3.2lf\\n");
	push(@tmp, "AREA:virus#EE44EE:Virus");
	push(@tmp, "GPRINT:virus:LAST:            Current\\: %3.2lf");
	push(@tmp, "GPRINT:virus:AVERAGE:  Average\\: %3.2lf");
	push(@tmp, "GPRINT:virus:MIN:  Min\\: %3.2lf");
	push(@tmp, "GPRINT:virus:MAX:  Max\\: %3.2lf\\n");
	push(@tmp, "AREA:n_delvd#4444EE:Delivered");
	push(@tmp, "GPRINT:delvd:LAST:        Current\\: %3.2lf");
	push(@tmp, "GPRINT:delvd:AVERAGE:  Average\\: %3.2lf");
	push(@tmp, "GPRINT:delvd:MIN:  Min\\: %3.2lf");
	push(@tmp, "GPRINT:delvd:MAX:  Max\\: %3.2lf\\n");
	push(@tmp, "AREA:n_out#44EEEE:Out Connections");
	push(@tmp, "GPRINT:out:LAST:  Current\\: %3.2lf");
	push(@tmp, "GPRINT:out:AVERAGE:  Average\\: %3.2lf");
	push(@tmp, "GPRINT:out:MIN:  Min\\: %3.2lf");
	push(@tmp, "GPRINT:out:MAX:  Max\\: %3.2lf\\n");
	push(@tmp, "LINE1:in#00EE00");
	push(@tmp, "LINE1:rejtd#EE0000");
	push(@tmp, "LINE1:recvd#1F881F");
	push(@tmp, "LINE1:spam#EEEE00");
	push(@tmp, "LINE1:virus#EE00EE");
	push(@tmp, "LINE1:n_delvd#0000EE");
	push(@tmp, "LINE1:n_out#00EEEE");

	push(@tmpz, "AREA:in#44EE44:In Connections");
	push(@tmpz, "AREA:rejtd#EE4444:Rejected");
	push(@tmpz, "AREA:recvd#448844:Received");
	push(@tmpz, "AREA:spam#EEEE44:Spam");
	push(@tmpz, "AREA:virus#EE44EE:Virus");
	push(@tmpz, "AREA:n_delvd#4444EE:Delivered");
	push(@tmpz, "AREA:n_out#44EEEE:Out Connections");
	push(@tmpz, "LINE1:in#00EE00");
	push(@tmpz, "LINE1:rejtd#EE0000");
	push(@tmpz, "LINE1:recvd#1F881F");
	push(@tmpz, "LINE1:spam#EEEE00");
	push(@tmpz, "LINE1:virus#EE00EE");
	push(@tmpz, "LINE1:n_delvd#0000EE");
	push(@tmpz, "LINE1:n_out#00EEEE");

	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}

	undef(@riglim);
	if($MAIL1_RIGID eq 1) {
		push(@riglim, "--upper-limit=$MAIL1_LIMIT");
	} else {
		if($MAIL1_RIGID eq 2) {
			push(@riglim, "--upper-limit=$MAIL1_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Mail  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Messages/sec",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:in=$MAIL_RRD:mail_in:AVERAGE",
		"DEF:out=$MAIL_RRD:mail_out:AVERAGE",
		"DEF:recvd=$MAIL_RRD:mail_recvd:AVERAGE",
		"DEF:delvd=$MAIL_RRD:mail_delvd:AVERAGE",
		"DEF:bytes_recvd=$MAIL_RRD:mail_bytes_recvd:AVERAGE",
		"DEF:bytes_delvd=$MAIL_RRD:mail_bytes_delvd:AVERAGE",
		"DEF:rejtd=$MAIL_RRD:mail_rejtd:AVERAGE",
		"DEF:spam=$MAIL_RRD:mail_spam:AVERAGE",
		"DEF:virus=$MAIL_RRD:mail_virus:AVERAGE",
		"CDEF:n_delvd=delvd,-1,*",
		"CDEF:n_out=out,-1,*",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Mail  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Messages/sec",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:in=$MAIL_RRD:mail_in:AVERAGE",
			"DEF:out=$MAIL_RRD:mail_out:AVERAGE",
			"DEF:recvd=$MAIL_RRD:mail_recvd:AVERAGE",
			"DEF:delvd=$MAIL_RRD:mail_delvd:AVERAGE",
			"DEF:bytes_recvd=$MAIL_RRD:mail_bytes_recvd:AVERAGE",
			"DEF:bytes_delvd=$MAIL_RRD:mail_bytes_delvd:AVERAGE",
			"DEF:rejtd=$MAIL_RRD:mail_rejtd:AVERAGE",
			"DEF:spam=$MAIL_RRD:mail_spam:AVERAGE",
			"DEF:virus=$MAIL_RRD:mail_virus:AVERAGE",
			"CDEF:n_delvd=delvd,-1,*",
			"CDEF:n_out=out,-1,*",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /mail1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@riglim);
	if($MAIL2_RIGID eq 1) {
		push(@riglim, "--upper-limit=$MAIL2_LIMIT");
	} else {
		if($MAIL2_RIGID eq 2) {
			push(@riglim, "--upper-limit=$MAIL2_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:queued#EEEE44:Queued");
	push(@tmp, "LINE1:queued#EEEE00");
	push(@tmp, "GPRINT:queued:LAST:          Current\\: %5.0lf\\n");
	push(@tmpz, "AREA:queued#EEEE44:Queued");
	push(@tmpz, "LINE1:queued#EEEE00");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	if($MAIL_SHOW_QUEUE eq "Y") {
		RRDs::graph("$PNG_DIR" . "$PNG2",
			"--title=Mail queue  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Messages",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:queued=$MAIL_RRD:mail_queued:AVERAGE",
			@tmp);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
		if($ENABLE_ZOOM eq "Y") {
			($width, $height) = split('x', $GRAPH_SIZE{zoom});
			RRDs::graph("$PNG_DIR" . "$PNG2z",
				"--title=Mail queue  (1$when)",
				"--start=-1$when",
				"--imgformat=PNG",
				"--vertical-label=Messages",
				"--width=$width",
				"--height=$height",
				@riglim,
				"--lower-limit=0",
				@VERSION12,
				@VERSION12_small,
				@graph_colors,
				"DEF:queued=$MAIL_RRD:mail_queued:AVERAGE",
				@tmpz);
			$err = RRDs::error;
			print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
		}
	} else {
		RRDs::graph("$PNG_DIR" . "$PNG2",
			"--title=Mail traffic  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=bytes/sec",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:bytes_recvd=$MAIL_RRD:mail_bytes_recvd:AVERAGE",
			"DEF:bytes_delvd=$MAIL_RRD:mail_bytes_delvd:AVERAGE",
			"CDEF:K_recvd=bytes_recvd,1024,/",
			"CDEF:K_delvd=bytes_delvd,1024,/",
			"AREA:bytes_recvd#44EE44:KB/s Received",
			"GPRINT:K_recvd:LAST:   Current\\: %5.0lf\\n",
			"AREA:bytes_delvd#4444EE:KB/s Delivered",
			"GPRINT:K_delvd:LAST:  Current\\: %5.0lf\\n",
			"AREA:bytes_delvd#4444EE:",
			"AREA:bytes_recvd#44EE44:",
			"LINE1:bytes_delvd#0000EE",
			"LINE1:bytes_recvd#00EE00");
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
		if($ENABLE_ZOOM eq "Y") {
			($width, $height) = split('x', $GRAPH_SIZE{zoom});
			RRDs::graph("$PNG_DIR" . "$PNG2z",
				"--title=Mail traffic  (1$when)",
				"--start=-1$when",
				"--imgformat=PNG",
				"--vertical-label=bytes/sec",
				"--width=$width",
				"--height=$height",
				@riglim,
				"--lower-limit=0",
				@VERSION12,
				@VERSION12_small,
				@graph_colors,
				"DEF:bytes_recvd=$MAIL_RRD:mail_bytes_recvd:AVERAGE",
				"DEF:bytes_delvd=$MAIL_RRD:mail_bytes_delvd:AVERAGE",
				"CDEF:K_recvd=bytes_recvd,1024,/",
				"CDEF:K_delvd=bytes_delvd,1024,/",
				"AREA:bytes_recvd#44EE44:KB/s Received",
				"AREA:bytes_delvd#4444EE:KB/s Delivered",
				"AREA:bytes_delvd#4444EE:",
				"AREA:bytes_recvd#44EE44:",
				"LINE1:bytes_delvd#0000EE",
				"LINE1:bytes_recvd#00EE00");
			$err = RRDs::error;
			print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
		}
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /mail2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:greylisted#4444EE:Greylisted");
	push(@tmp, "GPRINT:greylisted:LAST:      Current\\: %5.0lf\\n");
	push(@tmp, "AREA:whitelisted#44EEEE:Whitelisted");
	push(@tmp, "GPRINT:whitelisted:LAST:     Current\\: %5.0lf\\n");
	push(@tmp, "LINE1:greylisted#0000EE");
	push(@tmp, "LINE1:whitelisted#00EEEE");
	push(@tmp, "LINE1:records#EE0000:Records");
	push(@tmp, "GPRINT:records:LAST:         Current\\: %5.0lf\\n");
	push(@tmpz, "AREA:greylisted#4444EE:Greylisted");
	push(@tmpz, "AREA:whitelisted#44EEEE:Whitelisted");
	push(@tmpz, "LINE2:greylisted#0000EE");
	push(@tmpz, "LINE2:whitelisted#00EEEE");
	push(@tmpz, "LINE2:records#EE0000:Records");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=Greylisting  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Messages",
		"--width=$width",
		"--height=$height",
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:records=$MAIL_RRD:mail_val07:AVERAGE",
		"DEF:greylisted=$MAIL_RRD:mail_val08:AVERAGE",
		"DEF:whitelisted=$MAIL_RRD:mail_val09:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=Greylisting  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Messages",
			"--width=$width",
			"--height=$height",
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:records=$MAIL_RRD:mail_val07:AVERAGE",
			"DEF:greylisted=$MAIL_RRD:mail_val08:AVERAGE",
			"DEF:whitelisted=$MAIL_RRD:mail_val09:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /mail3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# PORT graph
# ----------------------------------------------------------------------------
sub port {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @riglim;
	my @warning;
	my @PNG;
	my @PNGz;
	my $addr;
	my $name;
	my @tmp;
	my @tmpz;
	my $n;
	my $str;
	my $err;

	my $PORT_PER_ROW = 3;
	$title = !$silent ? $title : "";

	if($IFACE_MODE eq "text") {
		my $line2;
		my $line3;
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$PORT_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		print("ERROR: while fetching $PORT_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		print("    ");
		for($n = 0; $n < $PORT_MAX && $n < scalar(@PORT_LIST); $n++) {
			printf("   %-5s %-8s", $PORT_LIST[$n], substr($PORT_NAME[$n], 0, 8));
			$line2 .= "   KB/s_I  KB/s_O";
			$line3 .= "-----------------";
		}
		print("\n");
		print("Time$line2\n");
		print("----$line3 \n");
		my $line;
		my @row;
		my $time;
		my $n2;
		my $from;
		my $to;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			$time = $time - (1 / $ts);
			printf(" %2d$tc ", $time);
			for($n2 = 0; $n2 < $PORT_MAX && $n2 < scalar(@PORT_LIST); $n2++) {
				$from = $n2 * 2;
				$to = $from + 1;
				my ($kin, $kout) = @$line[$from..$to];
#				$kin /= 1024;
#				$kout /= 1024;
				@row = ($kin, $kout);
				printf("  %6d  %6d ", @row);
			}
			print("\n");
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	for($n = 0; $n < $PORT_MAX; $n++) {
		$str = $u . $myself . $n . "." . $when . ".png";
		push(@PNG, $str);
		unlink("$PNG_DIR" . $str);
		if($ENABLE_ZOOM eq "Y") {
			$str = $u . $myself . $n . "z." . $when . ".png";
			push(@PNGz, $str);
			unlink("$PNG_DIR" . $str);
		}
	}

	$n = 0;
	while($n < $PORT_MAX) {
		if($title) {
			if($n == 0) {
				graph_header($title, $PORT_PER_ROW);
			}
			print("    <tr>\n");
		}
		for($n2 = 0; $n2 < $PORT_PER_ROW; $n2++) {
			last unless $n < $PORT_MAX;
			if($title) {
				print("    <td bgcolor='" . $title_bg_color . "'>\n");
			}
			undef(@riglim);
			if($PORT_RIGID[$n] eq 1) {
				push(@riglim, "--upper-limit=$PORT_LIMIT[$n]");
			} else {
				if($PORT_RIGID[$n] eq 2) {
					push(@riglim, "--upper-limit=$PORT_LIMIT[$n]");
					push(@riglim, "--rigid");
				}
			}

			undef(@warning);
			if($os eq "Linux") {
				open(IN, "netstat -nl --$PORT_PROT[$n] |");
				while(<IN>) {
					(undef, undef, undef, $addr) = split(' ', $_);
					chomp($addr);
					$addr =~ s/.*://;
					if($addr eq $PORT_LIST[$n]) {
						last;
					}
				}
				close(IN);
			} elsif($os eq "FreeBSD") {
				# this part must be finished!!!! XXX
				open(IN, "netstat -anl -p $PORT_PROT[$n] |");
				while(<IN>) {
				}
				close(IN);
			}
			if($addr ne $PORT_LIST[$n]) {
				push(@warning, $warning_color);
			}

			$name = substr($PORT_NAME[$n], 0, 8);
			undef(@tmp);
			undef(@tmpz);
			push(@tmp, "AREA:in#44EE44:Input");
			push(@tmp, "AREA:out#4444EE:Output");
			push(@tmp, "AREA:out#4444EE:");
			push(@tmp, "AREA:in#44EE44:");
			push(@tmp, "LINE1:out#0000EE");
			push(@tmp, "LINE1:in#00EE00");
			push(@tmpz, "AREA:in#44EE44:Input");
			push(@tmpz, "AREA:out#4444EE:Output");
			push(@tmpz, "AREA:out#4444EE:");
			push(@tmpz, "AREA:in#44EE44:");
			push(@tmpz, "LINE1:out#0000EE");
			push(@tmpz, "LINE1:in#00EE00");
			($width, $height) = split('x', $GRAPH_SIZE{mini});
			if($silent =~ /imagetag/) {
				($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
				($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
				push(@tmp, "COMMENT: \\n");
				push(@tmp, "COMMENT: \\n");
				push(@tmp, "COMMENT: \\n");
			}
			RRDs::graph("$PNG_DIR" . "$PNG[$n]",
				"--title=$name traffic  (1$when)",
				"--start=-1$when",
				"--imgformat=PNG",
				"--vertical-label=bytes/sec",
				"--width=$width",
				"--height=$height",
				@riglim,
				"--lower-limit=0",
				@VERSION12,
				@VERSION12_small,
				@graph_colors,
				@warning,
				"DEF:in=$PORT_RRD:port" . $n . "_in:AVERAGE",
				"DEF:out=$PORT_RRD:port" . $n . "_out:AVERAGE",
				@tmp);
			$err = RRDs::error;
			print("ERROR: while graphing $PNG_DIR" . "$PNG[$n]: $err\n") if $err;
			if($ENABLE_ZOOM eq "Y") {
				($width, $height) = split('x', $GRAPH_SIZE{zoom});
				RRDs::graph("$PNG_DIR" . "$PNGz[$n]",
					"--title=$name traffic  (1$when)",
					"--start=-1$when",
					"--imgformat=PNG",
					"--vertical-label=bytes/sec",
					"--width=$width",
					"--height=$height",
					@riglim,
					"--lower-limit=0",
					@VERSION12,
					@VERSION12_small,
					@graph_colors,
					@warning,
					"DEF:in=$PORT_RRD:port" . $n . "_in:AVERAGE",
					"DEF:out=$PORT_RRD:port" . $n . "_out:AVERAGE",
					@tmpz);
				$err = RRDs::error;
				print("ERROR: while graphing $$PNG_DIR" . "$PNGz[$n]: $err\n") if $err;
			}
			if($title || ($silent =~ /imagetag/ && $graph =~ /port$n/)) {
				if($ENABLE_ZOOM eq "Y") {
					print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNGz[$n] . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG[$n] . "' border='0'></a>\n");
				} else {
					print("      <img src='" . $URL . $IMGS_DIR . $PNG[$n] . "'>\n");
				}
			}
			if($title) {
				print("    </td>\n");
			}
			$n++;
		}
		if($title) {
			print("    </tr>\n");
		}
	}
	if($title) {
		graph_footer();
	}
	return 1;
}

# USER graph
# ----------------------------------------------------------------------------
sub user {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @riglim;
	my @tmp;
	my @tmpz;
	my $n;
	my $err;

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$USER_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		print("ERROR: while fetching $USER_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		print("Time   Telnet/SSH     Samba  Netatalk\n");
		print("------------------------------------- \n");
		my $line;
		my @row;
		my $time;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			my ($sys, $smb, $mac) = @$line;
			@row = ($sys, $smb, $mac);
			$time = $time - (1 / $ts);
			printf(" %2d$tc       %6d    %6d    %6d\n", $time, @row);
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}
	if($USER1_RIGID eq 1) {
		push(@riglim, "--upper-limit=$USER1_LIMIT");
	} else {
		if($USER1_RIGID eq 2) {
			push(@riglim, "--upper-limit=$USER1_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}
	push(@tmp, "AREA:sys#44EE44:Telnet/SSH");
	push(@tmp, "GPRINT:sys:LAST:       Current\\: %3.0lf");
	push(@tmp, "GPRINT:sys:AVERAGE:   Average\\: %3.0lf");
	push(@tmp, "GPRINT:sys:MIN:   Min\\: %3.0lf");
	push(@tmp, "GPRINT:sys:MAX:   Max\\: %3.0lf\\n");
	push(@tmp, "LINE1:sys#00EE00");
	push(@tmpz, "AREA:sys#44EE44:Telnet/SSH");
	push(@tmpz, "LINE1:sys#00EE00");
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Users logged on system  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Users",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:sys=$USER_RRD:user_sys:AVERAGE",
		"COMMENT: \\n",
		@tmp,
		"COMMENT: \\n",
		"COMMENT: \\n",
		"COMMENT: \\n");
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Users logged on system  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Users",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:sys=$USER_RRD:user_sys:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /user1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@riglim);
	if($USER2_RIGID eq 1) {
		push(@riglim, "--upper-limit=$USER2_LIMIT");
	} else {
		if($USER2_RIGID eq 2) {
			push(@riglim, "--upper-limit=$USER2_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:smb#EEEE44:Samba");
	push(@tmp, "GPRINT:smb:LAST:           Current\\: %3.0lf\\n");
	push(@tmp, "LINE1:smb#EEEE00");
	push(@tmpz, "AREA:smb#EEEE44:Samba");
	push(@tmpz, "LINE2:smb#EEEE00");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=Users using Samba  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Users",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:smb=$USER_RRD:user_smb:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=Users using Samba  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Users",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:smb=$USER_RRD:user_smb:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /user2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@riglim);
	if($USER3_RIGID eq 1) {
		push(@riglim, "--upper-limit=$USER3_LIMIT");
	} else {
		if($USER3_RIGID eq 2) {
			push(@riglim, "--upper-limit=$USER3_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:mac#EE4444:Netatalk");
	push(@tmp, "GPRINT:mac:LAST:        Current\\: %3.0lf\\n");
	push(@tmp, "LINE1:mac#EE0000");
	push(@tmpz, "AREA:mac#EE4444:Netatalk");
	push(@tmpz, "LINE2:mac#EE0000");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=Users using Netatalk  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Users",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:mac=$USER_RRD:user_mac:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=Users using Netatalk  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Users",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:mac=$USER_RRD:user_mac:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /user3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# APACHE graph
# ----------------------------------------------------------------------------
sub apache {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @riglim;
	my @tmp;
	my @tmpz;
	my $n;
	my $err;

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$APACHE_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		print("ERROR: while fetching $APACHE_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		print("Time   Acceses     kbytes      CPU  Busy  Idle\n");
		print("---------------------------------------------- \n");
		my $line;
		my @row;
		my $time;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			my ($acc, $kb, $cpu, $busy, $idle) = @$line;
			@row = ($acc, $kb, $cpu, $busy, $idle);
			$time = $time - (1 / $ts);
			printf(" %2d$tc   %7d  %9d    %4.2f%%   %3d   %3d\n", $time, @row);
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}
	if($APACHE1_RIGID eq 1) {
		push(@riglim, "--upper-limit=$APACHE1_LIMIT");
	} else {
		if($APACHE1_RIGID eq 2) {
			push(@riglim, "--upper-limit=$APACHE1_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}
	push(@tmp, "AREA:idle#4444EE:Idle");
	push(@tmp, "GPRINT:idle:LAST:          Current\\: %3.0lf");
	push(@tmp, "GPRINT:idle:AVERAGE:   Average\\: %3.0lf");
	push(@tmp, "GPRINT:idle:MIN:   Min\\: %3.0lf");
	push(@tmp, "GPRINT:idle:MAX:   Max\\: %3.0lf\\n");
	push(@tmp, "AREA:busy#44EEEE:Busy");
	push(@tmp, "GPRINT:busy:LAST:          Current\\: %3.0lf");
	push(@tmp, "GPRINT:busy:AVERAGE:   Average\\: %3.0lf");
	push(@tmp, "GPRINT:busy:MIN:   Min\\: %3.0lf");
	push(@tmp, "GPRINT:busy:MAX:   Max\\: %3.0lf\\n");
	push(@tmp, "LINE1:idle#0000EE");
	push(@tmp, "LINE1:busy#00EEEE");
	push(@tmp, "LINE1:tot#EE0000");
	push(@tmpz, "AREA:idle#4444EE:Idle");
	push(@tmpz, "AREA:busy#44EEEE:Busy");
	push(@tmpz, "LINE2:idle#0000EE");
	push(@tmpz, "LINE2:busy#00EEEE");
	push(@tmpz, "LINE2:tot#EE0000");
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Workers  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Workers",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:busy=$APACHE_RRD:apache_busy:AVERAGE",
		"DEF:idle=$APACHE_RRD:apache_idle:AVERAGE",
		"CDEF:tot=busy,idle,+",
		"COMMENT: \\n",
		@tmp,
		"COMMENT: \\n",
		"COMMENT: \\n");
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Workers  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Workers",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:busy=$APACHE_RRD:apache_busy:AVERAGE",
			"DEF:idle=$APACHE_RRD:apache_idle:AVERAGE",
			"CDEF:tot=busy,idle,+",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /apache1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@riglim);
	if($APACHE2_RIGID eq 1) {
		push(@riglim, "--upper-limit=$APACHE2_LIMIT");
	} else {
		if($APACHE2_RIGID eq 2) {
			push(@riglim, "--upper-limit=$APACHE2_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:cpu#44AAEE:CPU");
	push(@tmp, "GPRINT:cpu:LAST:             Current\\: %5.2lf%%\\n");
	push(@tmp, "LINE1:cpu#00EEEE");
	push(@tmpz, "AREA:cpu#44AAEE:CPU");
	push(@tmpz, "LINE1:cpu#00EEEE");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=CPU usage  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Percent (%)",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:cpu=$APACHE_RRD:apache_cpu:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=CPU usage  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Percent",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:cpu=$APACHE_RRD:apache_cpu:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /apache2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@riglim);
	if($APACHE3_RIGID eq 1) {
		push(@riglim, "--upper-limit=$APACHE3_LIMIT");
	} else {
		if($APACHE3_RIGID eq 2) {
			push(@riglim, "--upper-limit=$APACHE3_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:acc#44EE44:Accesses");
	push(@tmp, "GPRINT:acc:LAST:        Current\\: %5.2lf\\n");
	push(@tmp, "LINE1:acc#00EE00");
	push(@tmpz, "AREA:acc#44EE44:Accesses");
	push(@tmpz, "LINE1:acc#00EE00");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=Requests  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Accesses/s",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:acc=$APACHE_RRD:apache_acc:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=Requests  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Accesses/s",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:acc=$APACHE_RRD:apache_acc:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /apache3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# NGINX graph
# ----------------------------------------------------------------------------
sub nginx {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @riglim;
	my @warning;
	my @tmp;
	my @tmpz;
	my $addr;
	my $n;
	my $err;

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$NGINX_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		print("ERROR: while fetching $NGINX_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		print("Time   Total  Reading  Writing  Waiting Requests   KB/s_I   KB/s_O\n");
		print("------------------------------------------------------------------ \n");
		my $line;
		my @row;
		my $time;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			my ($req, $tot, $rea, $wri, $wai, $ki, $ko) = @$line;
			$ki /= 1024;
			$ko /= 1024;
			@row = ($tot, $rea, $wri, $wai, $req, $ki, $ko);
			$time = $time - (1 / $ts);
			printf(" %2d$tc  %6d   %6d   %6d   %6d   %6d   %6d   %6d\n", $time, @row);
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}
	if($NGINX1_RIGID eq 1) {
		push(@riglim, "--upper-limit=$NGINX1_LIMIT");
	} else {
		if($NGINX1_RIGID eq 2) {
			push(@riglim, "--upper-limit=$NGINX1_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}
	push(@tmp, "AREA:total#44EEEE:Total");
	push(@tmp, "GPRINT:total:LAST:          Current\\: %4.0lf");
	push(@tmp, "GPRINT:total:AVERAGE:    Average\\: %4.0lf");
	push(@tmp, "GPRINT:total:MIN:    Min\\: %4.0lf");
	push(@tmp, "GPRINT:total:MAX:    Max\\: %4.0lf\\n");
	push(@tmp, "AREA:reading#44EE44:Reading");
	push(@tmp, "GPRINT:reading:LAST:        Current\\: %4.0lf");
	push(@tmp, "GPRINT:reading:AVERAGE:    Average\\: %4.0lf");
	push(@tmp, "GPRINT:reading:MIN:    Min\\: %4.0lf");
	push(@tmp, "GPRINT:reading:MAX:    Max\\: %4.0lf\\n");
	push(@tmp, "AREA:writing#4444EE:Writing");
	push(@tmp, "GPRINT:writing:LAST:        Current\\: %4.0lf");
	push(@tmp, "GPRINT:writing:AVERAGE:    Average\\: %4.0lf");
	push(@tmp, "GPRINT:writing:MIN:    Min\\: %4.0lf");
	push(@tmp, "GPRINT:writing:MAX:    Max\\: %4.0lf\\n");
	push(@tmp, "AREA:waiting#EE44EE:Waiting");
	push(@tmp, "GPRINT:waiting:LAST:        Current\\: %4.0lf");
	push(@tmp, "GPRINT:waiting:AVERAGE:    Average\\: %4.0lf");
	push(@tmp, "GPRINT:waiting:MIN:    Min\\: %4.0lf");
	push(@tmp, "GPRINT:waiting:MAX:    Max\\: %4.0lf\\n");
	push(@tmp, "LINE1:total#00EEEE");
	push(@tmp, "LINE1:reading#00EE00");
	push(@tmp, "LINE1:writing#0000EE");
	push(@tmp, "LINE1:waiting#EE00EE");
	push(@tmpz, "AREA:total#44EEEE:Total");
	push(@tmpz, "AREA:reading#44EE44:Reading");
	push(@tmpz, "AREA:writing#4444EE:Writing");
	push(@tmpz, "AREA:waiting#EE44EE:Waiting");
	push(@tmpz, "LINE1:total#00EEEE");
	push(@tmpz, "LINE1:reading#00EE00");
	push(@tmpz, "LINE1:writing#0000EE");
	push(@tmpz, "LINE1:waiting#EE00EE");
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Connections  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Connections/s",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		"DEF:total=$NGINX_RRD:nginx_total:AVERAGE",
		"DEF:reading=$NGINX_RRD:nginx_reading:AVERAGE",
		"DEF:writing=$NGINX_RRD:nginx_writing:AVERAGE",
		"DEF:waiting=$NGINX_RRD:nginx_waiting:AVERAGE",
		@tmp,
		"COMMENT: \\n");
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Connections  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Connections/s",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:total=$NGINX_RRD:nginx_total:AVERAGE",
			"DEF:reading=$NGINX_RRD:nginx_reading:AVERAGE",
			"DEF:writing=$NGINX_RRD:nginx_writing:AVERAGE",
			"DEF:waiting=$NGINX_RRD:nginx_waiting:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /nginx1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@riglim);
	if($NGINX2_RIGID eq 1) {
		push(@riglim, "--upper-limit=$NGINX2_LIMIT");
	} else {
		if($NGINX2_RIGID eq 2) {
			push(@riglim, "--upper-limit=$NGINX2_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:requests#44EEEE:Requests");
	push(@tmp, "GPRINT:requests:LAST:        Current\\: %5.1lf\\n");
	push(@tmp, "LINE1:requests#00EEEE");
	push(@tmpz, "AREA:requests#44EEEE:Requests");
	push(@tmpz, "LINE1:requests#00EEEE");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=Requests  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Requests/s",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		"DEF:requests=$NGINX_RRD:nginx_requests:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=Requests  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Requests/s",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			"DEF:requests=$NGINX_RRD:nginx_requests:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /nginx2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@warning);
	if($os eq "Linux") {
		open(IN, "netstat -nl --tcp |");
		while(<IN>) {
			(undef, undef, undef, $addr) = split(' ', $_);
			chomp($addr);
			$addr =~ s/.*://;
			if($addr eq $NGINX_PORT) {
				last;
			}
		}
		close(IN);
	} elsif($os eq "FreeBSD") {
		# this part must be finished!!!! XXX
		open(IN, "netstat -anl -p tcp |");
		while(<IN>) {
		}
		close(IN);
	}
	if($addr ne $NGINX_PORT) {
		push(@warning, $warning_color);
	}

	undef(@riglim);
	if($NGINX3_RIGID eq 1) {
		push(@riglim, "--upper-limit=$NGINX3_LIMIT");
	} else {
		if($NGINX3_RIGID eq 2) {
			push(@riglim, "--upper-limit=$NGINX3_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	undef(@tmp);
	undef(@tmpz);
	push(@tmp, "AREA:in#44EE44:Input");
	push(@tmp, "AREA:out#4444EE:Output");
	push(@tmp, "AREA:out#4444EE:");
	push(@tmp, "AREA:in#44EE44:");
	push(@tmp, "LINE1:out#0000EE");
	push(@tmp, "LINE1:in#00EE00");
	push(@tmpz, "AREA:in#44EE44:Input");
	push(@tmpz, "AREA:out#4444EE:Output");
	push(@tmpz, "AREA:out#4444EE:");
	push(@tmpz, "AREA:in#44EE44:");
	push(@tmpz, "LINE1:out#0000EE");
	push(@tmpz, "LINE1:in#00EE00");
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
		@tmp = @tmpz;
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
		push(@tmp, "COMMENT: \\n");
	}
	RRDs::graph("$PNG_DIR" . "$PNG3",
		"--title=Traffic  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=bytes/s",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		@warning,
		"DEF:in=$NGINX_RRD:nginx_bytes_in:AVERAGE",
		"DEF:out=$NGINX_RRD:nginx_bytes_out:AVERAGE",
		@tmp);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG3z",
			"--title=Traffic  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=bytes/s",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			@warning,
			"DEF:in=$NGINX_RRD:nginx_bytes_in:AVERAGE",
			"DEF:out=$NGINX_RRD:nginx_bytes_out:AVERAGE",
			@tmpz);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /nginx3/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
		}
	}

	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# INT graph
# ----------------------------------------------------------------------------
sub int {
	my ($myself, $title) = @_;

	my $width;
	my $height;
	my @riglim;
	my $n;
	my $err;

	my @INT;
	my @NAME;
	my @DEF1;
	my @AREA1;
	my @LINE1;
	my @DEF2;
	my @AREA2;
	my @LINE2;
	my @DEF3;
	my @AREA3;
	my @LINE3;
	my $n1;
	my $n2;
	my $n3;
	my @ACOLOR1 =  ("#FFA500",
			"#44EEEE",
			"#CCCCCC",
			"#B4B444",
			"#4444EE",
			"#44EE44",
			"#EEEE44",
			"#444444",
			"#EE44EE",
			"#EE4444",
			"#448844",
			"#BB44EE",
			"#D3D701",
			"#E29136",
			"#DDAE8C",
			"#F29967",
			"#996952",
			"#EB6C75",
			"#B84F6B",
			"#963C74",
			"#A68BBC",
			"#597AB7",
			"#8CB4CE",
			"#63BEE0",
			"#3CB5B0",
			"#7EB97C",
			"#94C36B",
			"#884632");
	my @LCOLOR1 =  ("#DDA500",
			"#00EEEE",
			"#888888",
			"#B4B400",
			"#0000EE",
			"#00EE00",
			"#EEEE00",
			"#444444",
			"#EE00EE",
			"#EE0000",
			"#008800",
			"#BB00EE",
			"#C8D200",
			"#DB6612",
			"#CE8248",
			"#EB6A39",
			"#8F4C30",
			"#E20045",
			"#B50C51",
			"#7B0059",
			"#684894",
			"#125AA3",
			"#518FBA",
			"#00AADA",
			"#009790",
			"#359B52",
			"#56AB27",
			"#782F1E");

	my @ACOLOR2 =  ("#44EEEE",
			"#44EE44",
			"#EE4444");
	my @LCOLOR2 =  ("#00EEEE",
			"#00EE00",
			"#EE0000");

	my @ACOLOR3 =  ("#44EE44",
			"#4444EE",
			"#EE44EE");
	my @LCOLOR3 =  ("#00EE00",
			"#0000EE",
			"#EE00EE");

	my $PNG1 = $u . $myself . "1." . $when . ".png";
	my $PNG2 = $u . $myself . "2." . $when . ".png";
	my $PNG3 = $u . $myself . "3." . $when . ".png";
	my $PNG1z = $u . $myself . "1z." . $when . ".png";
	my $PNG2z = $u . $myself . "2z." . $when . ".png";
	my $PNG3z = $u . $myself . "3z." . $when . ".png";

	$title = !$silent ? $title : "";

	unlink ("$PNG_DIR" . "$PNG1",
		"$PNG_DIR" . "$PNG2",
		"$PNG_DIR" . "$PNG3");
	if($ENABLE_ZOOM eq "Y") {
		unlink ("$PNG_DIR" . "$PNG1z",
			"$PNG_DIR" . "$PNG2z",
			"$PNG_DIR" . "$PNG3z");
	}

	if($os eq "Linux") {
		open(IN, "/proc/interrupts");
		my $timer_pos = 0;
		my $i8042_pos = 0;
		my $good_pos = 0;
		my $num;
		my $name;
		while(<IN>) {
			if(/[0-9]:/) {
				# Assuming int 0 will be only for timer
				if(/\s+0:/) {
					$timer_pos = index($_, "timer", 0);
				}
				# Assuming int 1 will be only for i8042
				if(/\s+1:/) {
					$i8042_pos = index($_, "i8042", 0);
				}
				$timer_pos = $timer_pos == 0 ? 999 : $timer_pos;
				$i8042_pos = $i8042_pos == -1 ? 0 : $i8042_pos;
				$good_pos = $timer_pos > $i8042_pos ? $i8042_pos : $timer_pos;
				$good_pos = $good_pos ? $good_pos : $timer_pos;
				$num = unpack("A4", $_);
				undef($name);
				if(length($_) >= $good_pos) {
					$name = substr($_, $good_pos);
					$name = defined($name) ? $name : "";
				}
				chomp($num, $name);
				$name =~ s/^\s+//;
				$num =~ s/^\s+//;
				$num =~ s/:.*//;
				$n = $num;
				$num = $num > 255 ? $num % 256 : $num;
				$INT[$num] = defined($INT[$num]) ? $INT[$num] . "," : "";
				$NAME[$num] = defined($NAME[$num]) ? $NAME[$num] . ", " : "";
				$INT[$num] .= $n;
				$NAME[$num] .= $name;
			}
		}
		close(IN);
	} elsif ($os eq "FreeBSD") {
		open(IN, "vmstat -i | sort |");
		my @allfields;
		my $num;
		my $name;
		while(<IN>) {
			if(/^\D{3}\d+/) {
				@allfields = split(' ', $_);
				$num = $allfields[0];
				$name = "";
				for($n = 1; $n <= $#allfields - 2; $n++) {
					$name .= $allfields[$n] . " ";
				}
				$num =~ s/^\D{3}//;
				$num =~ s/://;
				$name =~ s/\s+$//;

				# only the first timer (cpu0) is covered
				if($name eq "timer") {
					if($num != 0) {
						next;
					}
				}

				$n = $num;
				$num = $num > 255 ? $num % 256 : $num;
				$INT[$num] = defined($INT[$num]) ? $INT[$num] . "," : "";
				$NAME[$num] = defined($NAME[$num]) ? $NAME[$num] . ", " : "";
				$INT[$num] .= $n;
				$NAME[$num] .= $name;
			}
		}
		close(IN);

		chomp(@NAME);
		# strip all blank spaces at the end of the strings
		for($n = 0; $n < 256; $n++) {
			if(defined($NAME[$n])) {
				$NAME[$n] =~ s/\s+$//;
			}
		}
	}

	if($IFACE_MODE eq "text") {
		if($title) {
			graph_header($title, 2);
			print("    <tr>\n");
			print("    <td bgcolor='$title_bg_color'>\n");
		}
		my (undef, undef, undef, $data) = RRDs::fetch("$INT_RRD",
			"--start=-1$when",
			"AVERAGE",
			"-r $res");
		$err = RRDs::error;
		my $line1;
		print("ERROR: while fetching $INT_RRD: $err\n") if $err;
		print("    <pre style='font-size: 12px; color: $fg_color';>\n");
		print("Time   ");
		for($n = 0; $n < 256; $n++) {
			if(defined($INT[$n])) {
				printf(" %8s", $INT[$n]);
				$line1 .= "---------";
			}
		}
		print(" \n");
		print("-------$line1\n");
		my $line;
		my @row;
		my $time;
		for($n = 0, $time = $tb; $n < ($tb * $ts); $n++) {
			$line = @$data[$n];
			@row = @$line;
			$time = $time - (1 / $ts);
			printf(" %2d$tc   ", $time);
			for($n2 = 0; $n2 < 256; $n2++) {
				if(defined($INT[$n2])) {
					printf(" %8d", $row[$n2]);
				}
			}
			print(" \n");
		}
		print("    </pre>\n");
		if($title) {
			print("    </td>\n");
			print("    </tr>\n");
			graph_footer();
		}
		return 1;
	}

	if($title) {
		graph_header($title, 2);
	}

	my $i;
	for($n = 0, $n1 = 0, $n2 = 0, $n3 = 0; $n < 256; $n++) {
		if(defined($NAME[$n])) {
			# We need to escape colons to support RRDtool v1.2+
			if($RRDs::VERSION > 1.2) {
				$NAME[$n] =~ s/:/\\:/g;
			}
		}
		if(defined($INT[$n])) {
			if(index($INT[$n], ",", 0) < 0) {
				$i = $INT[$n];
			} else {
				($i) = split(',', $INT[$n]);
			}
			if($i < 3) {
				push(@DEF2, ("DEF:int" . $n . "=" . $INT_RRD . ":int_" . $n . ":AVERAGE"));
				push(@AREA2, ("AREA:int" . $n . $ACOLOR2[$n2] . ":(" . $INT[$n] . ")" . $NAME[$n]));
				push(@LINE2, ("LINE1:int" . $n . $LCOLOR2[$n2]));
				$n2++;
			} elsif($i < 6) {
				push(@DEF3, ("DEF:int" . $n . "=" . $INT_RRD . ":int_" . $n . ":AVERAGE"));
				push(@AREA3, ("AREA:int" . $n . $ACOLOR3[$n3] . ":(" . $INT[$n] . ")" . $NAME[$n]));
				push(@LINE3, ("LINE1:int" . $n . $LCOLOR3[$n3]));
				$n3++;
			} else {
				push(@DEF1, ("DEF:int" . $n . "=" . $INT_RRD . ":int_" . $n . ":AVERAGE"));
				push(@AREA1, ("AREA:int" . $n . $ACOLOR1[$n1] . ":(" . $INT[$n] . ")" . $NAME[$n]));
				push(@LINE1, ("LINE1:int" . $n . $LCOLOR1[$n1]));
				$n1++;
				if(!($n1 % 3)) {
					push(@AREA1, ("COMMENT: \\n"));
				}
			}
		}
	}
	push(@AREA1, ("COMMENT: \\n"));
	if($INT1_RIGID eq 1) {
		push(@riglim, "--upper-limit=$INT1_LIMIT");
	} else {
		if($INT1_RIGID eq 2) {
			push(@riglim, "--upper-limit=$INT1_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	if($title) {
		print("    <tr>\n");
		print("    <td bgcolor='$title_bg_color'>\n");
	}
	($width, $height) = split('x', $GRAPH_SIZE{main});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
	}
	RRDs::graph("$PNG_DIR" . "$PNG1",
		"--title=Interrupt activity  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Ticks/sec",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@graph_colors,
		@DEF1,
		@AREA1,
		@LINE1);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG1: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG1z",
			"--title=Interrupt activity  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Ticks/sec",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			@DEF1,
			@AREA1,
			@LINE1);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG1z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /int1/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG1z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG1 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG1 . "'>\n");
		}
	}
	if($title) {
		print("    </td>\n");
		print("    <td valign='top' bgcolor='" . $title_bg_color . "'>\n");
	}

	undef(@riglim);
	if($INT2_RIGID eq 1) {
		push(@riglim, "--upper-limit=$INT2_LIMIT");
	} else {
		if($INT2_RIGID eq 2) {
			push(@riglim, "--upper-limit=$INT2_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
	}
	RRDs::graph("$PNG_DIR" . "$PNG2",
		"--title=Core activity  (1$when)",
		"--start=-1$when",
		"--imgformat=PNG",
		"--vertical-label=Ticks/sec",
		"--width=$width",
		"--height=$height",
		@riglim,
		"--lower-limit=0",
		@VERSION12,
		@VERSION12_small,
		@graph_colors,
		@DEF2,
		@AREA2,
		@LINE2);
	$err = RRDs::error;
	print("ERROR: while graphing $PNG_DIR" . "$PNG2: $err\n") if $err;
	if($ENABLE_ZOOM eq "Y") {
		($width, $height) = split('x', $GRAPH_SIZE{zoom});
		RRDs::graph("$PNG_DIR" . "$PNG2z",
			"--title=Core activity  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Ticks/sec",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			@DEF2,
			@AREA2,
			@LINE2);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG2z: $err\n") if $err;
	}
	if($title || ($silent =~ /imagetag/ && $graph =~ /int2/)) {
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG2z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG2 . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG2 . "'>\n");
		}
	}

	undef(@riglim);
	if($INT3_RIGID eq 1) {
		push(@riglim, "--upper-limit=$INT3_LIMIT");
	} else {
		if($INT3_RIGID eq 2) {
			push(@riglim, "--upper-limit=$INT3_LIMIT");
			push(@riglim, "--rigid");
		}
	}
	($width, $height) = split('x', $GRAPH_SIZE{small});
	if($silent =~ /imagetag/) {
		($width, $height) = split('x', $GRAPH_SIZE{remote}) if $silent eq "imagetag";
		($width, $height) = split('x', $GRAPH_SIZE{main}) if $silent eq "imagetagbig";
	}
	if(@DEF3 && @AREA3 && @LINE3) {
		RRDs::graph("$PNG_DIR" . "$PNG3",
			"--title=Low-devices activity  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=Ticks/sec",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@VERSION12_small,
			@graph_colors,
			@DEF3,
			@AREA3,
			@LINE3);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG3: $err\n") if $err;
		if($ENABLE_ZOOM eq "Y") {
			($width, $height) = split('x', $GRAPH_SIZE{zoom});
			RRDs::graph("$PNG_DIR" . "$PNG3z",
				"--title=Low-devices activity  (1$when)",
				"--start=-1$when",
				"--imgformat=PNG",
				"--vertical-label=Ticks/sec",
				"--width=$width",
				"--height=$height",
				@riglim,
				"--lower-limit=0",
				@VERSION12,
				@VERSION12_small,
				@graph_colors,
				@DEF3,
				@AREA3,
				@LINE3);
			$err = RRDs::error;
			print("ERROR: while graphing $PNG_DIR" . "$PNG3z: $err\n") if $err;
		}
		if($title || ($silent =~ /imagetag/ && $graph =~ /int3/)) {
			if($ENABLE_ZOOM eq "Y") {
				print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNG3z . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG3 . "' border='0'></a>\n");
			} else {
				print("      <img src='" . $URL . $IMGS_DIR . $PNG3 . "'>\n");
			}
		}
	}
	if($title) {
		print("    </td>\n");
		print("    </tr>\n");
		graph_footer();
	}
	return 1;
}

# Multihost
# ----------------------------------------------------------------------------
sub multihost {
	my $n;
	my $n2;
	my @HOST;
	my @URL;

	for($n = 0; $n < scalar(@REMOTEHOST_LIST); $n += 2) {
		push(@HOST, $REMOTEHOST_LIST[$n]);
		push(@URL, $REMOTEHOST_LIST[$n + 1]);
	}

	$MULTIHOST_IMGS_PER_ROW = 1 unless $MULTIHOST_IMGS_PER_ROW > 1;
	$graph = $graph eq "all" ? "_system1" : $graph;

	if($val eq "all") {
		for($n = 0; $n < scalar(@HOST); $n += $MULTIHOST_IMGS_PER_ROW) {
			print "<table cellspacing='5' cellpadding='0' width='1' bgcolor='$graph_bg_color' border='1'>\n";
			print " <tr>\n";
			for($n2 = 0; $n2 < $MULTIHOST_IMGS_PER_ROW; $n2++) {
				if($n < scalar(@HOST)) {
					print "  <td bgcolor='$title_bg_color'>\n";
					print "   <font face='Verdana, Tahoma, sans-serif' color='$fg_color'>\n";
					print "   <b>&nbsp;&nbsp;" . $HOST[$n] . "<b>\n";
					print "   </font>\n";
					print "  </td>\n";
				}
				$n++;
			}
			print " </tr>\n";
			print " <tr>\n";
			for($n2 = 0, $n = $n - $MULTIHOST_IMGS_PER_ROW; $n2 < $MULTIHOST_IMGS_PER_ROW; $n2++) {
				if($n < scalar(@HOST)) {
					print "  <td bgcolor='$title_bg_color' style='vertical-align: top; height: 10%; width: 10%;'>\n";
					print "   <iframe src=$URL[$n]$BASE_CGI/monitorix.cgi?mode=localhost&when=$when&graph=$graph&color=$color&silent=imagetag height=201 width=397 frameborder=0 marginwidth=0 marginheight=0 scrolling=no></iframe>\n";
					print "  </td>\n";

				}
				$n++;
			}
			print " </tr>\n";
			print " <tr>\n";
			for($n2 = 0, $n = $n - $MULTIHOST_IMGS_PER_ROW; $n2 < $MULTIHOST_IMGS_PER_ROW; $n2++) {
				if($n < scalar(@HOST)) {
				if($MULTIHOST_FOOTER) {
					print "  <td bgcolor='$title_bg_color'>\n";
					print "   <font face='Verdana, Tahoma, sans-serif' color='$title_fg_color'>\n";
					print "   <font size='-1'>\n";
					print "    <b>&nbsp;&nbsp;<a href='" . $URL[$n] . $BASE_URL . "/' style='{color: $title_fg_color}'>$URL[$n]</a><b>\n";
					print "   </font></font>\n";
					print "  </td>\n";
				}
				}
				$n++;
			}
			$n = $n - $MULTIHOST_IMGS_PER_ROW;
			print " </tr>\n";
			print "</table>\n";
			print "<br>\n";
		}
	} else {
		print "  <table cellspacing='5' cellpadding='0' width='1' bgcolor='$graph_bg_color' border='1'>\n";
		print "   <tr>\n";
		print "    <td bgcolor='$title_bg_color'>\n";
		print "    <font face='Verdana, Tahoma, sans-serif' color='$fg_color'>\n";
		print "    <b>&nbsp;&nbsp;" . $HOST[$val] . "<b>\n";
		print "    </font>\n";
		print "    </td>\n";
		print "   </tr>\n";
		print "   <tr>\n";
		print "    <td bgcolor='$title_bg_color' style='vertical-align: top; height: 10%; width: 10%;'>\n";
		print "     <iframe src=$URL[$val]$BASE_CGI/monitorix.cgi?mode=localhost&when=$when&graph=$graph&color=$color&silent=imagetagbig height=249 width=545 frameborder=0 marginwidth=0 marginheight=0 scrolling=no></iframe>\n";
		print "    </td>\n";
		print "   </tr>\n";
		print "   <tr>\n";
		if($MULTIHOST_FOOTER) {
			print "   <td bgcolor='$title_bg_color'>\n";
			print "    <font face='Verdana, Tahoma, sans-serif' color='$title_fg_color'>\n";
			print "    <font size='-1'>\n";
			print "    <b>&nbsp;&nbsp;<a href='" . $URL[$val] . "/monitorix/' style='{color: $title_fg_color}'>$URL[$val]</a><b>\n";
			print "    </font></font>\n";
			print "   </td>\n";
		}
		print "   </tr>\n";
		print "  </table>\n";
		print "  <br>\n";
	}
}

sub pc {
	my $width;
	my $height;
	my @riglim;
	my @PNG;
	my @PNGz;
	my @tmp;
	my @tmpz;
	my $n;
	my $n2;
	my $str;
	my $err;

	for($n = 0; $n < $PC_MAX; $n++) {
		$str = $u . "pc" . $n . ".$when" . ".png";
		push(@PNG, $str);
		unlink("$PNG_DIR" . $str);
		if($ENABLE_ZOOM eq "Y") {
			$str = $u . "pc" . $n . "z.$when" . ".png";
			push(@PNGz, $str);
			unlink("$PNG_DIR" . $str);
		}
	}
	if($PC_RIGID eq 1) {
		$riglim[0] = "--upper-limit=$PC_LIMIT";
	} else {
		if($PC_RIGID eq 2) {
			$riglim[0] = "--upper-limit=$PC_LIMIT";
			$riglim[1] = "--rigid";
		}
	}

	$PC_IMGS_PER_ROW = 1 unless $PC_IMGS_PER_ROW > 1;
	if($val eq "all") {
		print("  <table cellspacing='5' cellpadding='0' width='1' bgcolor='$graph_bg_color' border='1'>\n");
		print("  <tr>\n");
		print("  <td bgcolor='$title_bg_color' colspan='" . $PC_IMGS_PER_ROW  . "'>\n");
		print("  <font face='Verdana, Tahoma, sans-serif' color='$title_fg_color'>\n");
		print("    <b>&nbsp;&nbsp;Internet traffic and usage<b>\n");
		print("  </font>\n");
		print("  </td>\n");
		print("  </tr>\n");
		$n = 0;
		while($n < $PC_MAX) {
			last unless $PC_LIST[$n];
			print("  <tr>\n");
			for($n2 = 0; $n2 < $PC_IMGS_PER_ROW; $n2++) {
				last unless $n < $PC_MAX;
				print("  <td bgcolor='$title_bg_color'>\n");
				undef(@tmp);
				undef(@tmpz);
				push(@tmp, "AREA:in#44EE44:Input");
				push(@tmp, "AREA:out#4444EE:Output");
				push(@tmp, "AREA:out#4444EE:");
				push(@tmp, "AREA:in#44EE44:");
				push(@tmp, "LINE1:out#0000EE");
				push(@tmp, "LINE1:in#00EE00");
				push(@tmpz, "AREA:in#44EE44:Input");
				push(@tmpz, "AREA:out#4444EE:Output");
				push(@tmpz, "AREA:out#4444EE:");
				push(@tmpz, "AREA:in#44EE44:");
				push(@tmpz, "LINE1:out#0000EE");
				push(@tmpz, "LINE1:in#00EE00");
				($width, $height) = split('x', $GRAPH_SIZE{remote});
				RRDs::graph("$PNG_DIR" . "$PNG[$n]",
					"--title=$PC_LIST[$n] traffic  (1$when)",
					"--start=-1$when",
					"--imgformat=PNG",
					"--vertical-label=bytes/sec",
					"--width=$width",
					"--height=$height",
					@riglim,
					"--lower-limit=0",
					@VERSION12,
					@VERSION12_small,
					@graph_colors,
					"DEF:in=$PC_RRD:pc" . $n . "_in:AVERAGE",
					"DEF:out=$PC_RRD:pc" . $n . "_out:AVERAGE",
					@tmp);
				$err = RRDs::error;
				print("ERROR: while graphing $PNG_DIR" . "$PNG[$n]: $err\n") if $err;
				if($ENABLE_ZOOM eq "Y") {
					($width, $height) = split('x', $GRAPH_SIZE{zoom});
					RRDs::graph("$PNG_DIR" . "$PNGz[$n]",
						"--title=$PC_LIST[$n] traffic  (1$when)",
						"--start=-1$when",
						"--imgformat=PNG",
						"--vertical-label=bytes/sec",
						"--width=$width",
						"--height=$height",
						@riglim,
						"--lower-limit=0",
						@VERSION12,
						@VERSION12_small,
						@graph_colors,
						"DEF:in=$PC_RRD:pc" . $n . "_in:AVERAGE",
						"DEF:out=$PC_RRD:pc" . $n . "_out:AVERAGE",
						@tmpz);
					$err = RRDs::error;
					print("ERROR: while graphing $PNG_DIR" . "$PNGz[$n]: $err\n") if $err;
				}
				if($ENABLE_ZOOM eq "Y") {
					print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNGz[$n] . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG[$n] . "' border='0'></a>\n");
				} else {
					print("      <img src='" . $URL . $IMGS_DIR . $PNG[$n] . "'>\n");
				}
				print("  </td>\n");
				$n++;
			}
			print("  </tr>\n");
		}
		print "  </table>\n";
	} else {
		return unless $PC_LIST[$val];
		if(!$silent) {
			print("  <table cellspacing='5' cellpadding='0' width='1' bgcolor='$graph_bg_color' border='1'>\n");
			print("  <tr>\n");
			print("  <td bgcolor='$title_bg_color' colspan='1'>\n");
			print("  <font face='Verdana, Tahoma, sans-serif' color='$title_fg_color'>\n");
			print("    <b>&nbsp;&nbsp;Internet traffic and usage<b>\n");
			print("  </font>\n");
			print("  </td>\n");
			print("  </tr>\n");
			print("  <tr>\n");
			print("  <td bgcolor='$title_bg_color'>\n");
		}
		undef(@tmp);
		undef(@tmpz);
		push(@tmp, "AREA:B_in#44EE44:KB/s Input");
		push(@tmp, "GPRINT:K_in:LAST:     Current\\: %5.0lf");
		push(@tmp, "GPRINT:K_in:AVERAGE: Average\\: %5.0lf");
		push(@tmp, "GPRINT:K_in:MIN:    Min\\: %5.0lf");
		push(@tmp, "GPRINT:K_in:MAX:    Max\\: %5.0lf\\n");
		push(@tmp, "AREA:B_out#4444EE:KB/s Output");
		push(@tmp, "GPRINT:K_out:LAST:    Current\\: %5.0lf");
		push(@tmp, "GPRINT:K_out:AVERAGE: Average\\: %5.0lf");
		push(@tmp, "GPRINT:K_out:MIN:    Min\\: %5.0lf");
		push(@tmp, "GPRINT:K_out:MAX:    Max\\: %5.0lf\\n");
		push(@tmp, "AREA:B_out#4444EE:");
		push(@tmp, "AREA:B_in#44EE44:");
		push(@tmp, "LINE1:B_out#0000EE");
		push(@tmp, "LINE1:B_in#00EE00");
		push(@tmpz, "AREA:B_in#44EE44:Input");
		push(@tmpz, "AREA:B_out#4444EE:Output");
		push(@tmpz, "AREA:B_out#4444EE:");
		push(@tmpz, "AREA:B_in#44EE44:");
		push(@tmpz, "LINE1:B_out#0000EE");
		push(@tmpz, "LINE1:B_in#00EE00");
		($width, $height) = split('x', $GRAPH_SIZE{main});
		RRDs::graph("$PNG_DIR" . "$PNG[$val]",
			"--title=$PC_LIST[$val] traffic  (1$when)",
			"--start=-1$when",
			"--imgformat=PNG",
			"--vertical-label=bytes/sec",
			"--width=$width",
			"--height=$height",
			@riglim,
			"--lower-limit=0",
			@VERSION12,
			@graph_colors,
			"DEF:B_in=$PC_RRD:pc" . $val . "_in:AVERAGE",
			"DEF:B_out=$PC_RRD:pc" . $val . "_out:AVERAGE",
			"CDEF:K_in=B_in,1024,/",
			"CDEF:K_out=B_out,1024,/",
			@tmp);
		$err = RRDs::error;
		print("ERROR: while graphing $PNG_DIR" . "$PNG[$val]: $err\n") if $err;
		if($ENABLE_ZOOM eq "Y") {
			($width, $height) = split('x', $GRAPH_SIZE{zoom});
			RRDs::graph("$PNG_DIR" . "$PNGz[$val]",
				"--title=$PC_LIST[$val] traffic  (1$when)",
				"--start=-1$when",
				"--imgformat=PNG",
				"--vertical-label=bytes/sec",
				"--width=$width",
				"--height=$height",
				@riglim,
				"--lower-limit=0",
				@VERSION12,
				@graph_colors,
				"DEF:B_in=$PC_RRD:pc" . $val . "_in:AVERAGE",
				"DEF:B_out=$PC_RRD:pc" . $val . "_out:AVERAGE",
				"CDEF:K_in=B_in,1024,/",
				"CDEF:K_out=B_out,1024,/",
				@tmpz);
			$err = RRDs::error;
			print("ERROR: while graphing $PNG_DIR" . "$PNGz[$val]: $err\n") if $err;
		}
		if($ENABLE_ZOOM eq "Y") {
			print("      <a href=\"javascript:void(window.open('" . $URL . $IMGS_DIR . $PNGz[$val] . "','','width=" . ($width + 115) . ",height=" . ($height + 100) . ",scrollbars=0,resizable=0'))\"><img src='" . $URL . $IMGS_DIR . $PNG[$val] . "' border='0'></a>\n");
		} else {
			print("      <img src='" . $URL . $IMGS_DIR . $PNG[$val] . "'>\n");
		}
		if(!$silent) {
			print("  </td>\n");
			print "  </td>\n";
			print "  </tr>\n";
			print "  </table>\n";
		}
	}
}

sub graph_header {
	my ($title, $colspan) = @_;
	print("\n");
	print("  <table cellspacing='5' cellpadding='0' width='1' bgcolor='$graph_bg_color' border='1'>\n");
	print("    <tr>\n");
	print("      <td bgcolor='$title_bg_color' colspan='$colspan'>\n");
	print("        <font face='Verdana, Tahoma, sans-serif' color='$title_fg_color'>\n");
	print("          <b>&nbsp;&nbsp;$title<b>\n");
	print("        </font>\n");
	print("      </td>\n");
	print("    </tr>\n");
}

sub graph_footer {
	print("  </table>\n");
}


# MAIN
# ----------------------------------------------------------------------------
print("Content-Type: text/html\n");
print("\n");
if(!$silent) {
	my $title;
	my $str;

	print("<html>\n");
	print("  <head>\n");
	print("    <title>$TITLE</title>\n");
	print("    <link rel='shortcut icon' href='" . $URL . "monitorixico.png'>\n");
	print("    <meta http-equiv='Refresh' content='" . $REFRESH_RATE . "'>\n");
	print("  </head>\n");
	print("  <body bgcolor='" . $bg_color . "' vlink='#888888' link='#888888'>\n");
	print("  <center>\n");
	print("  <table cellspacing='5' cellpadding='0' bgcolor='" . $graph_bg_color . "' border='1'>\n");
	print("  <tr>\n");
	if($val ne "all" && $mode ne "multihost") {
		print("  <td bgcolor='" . $title_bg_color . "'>\n");
		print("  <font face='Verdana, Tahoma, sans-serif' color='" . $title_fg_color . "'>\n");
		print("    <font size='5'><b>&nbsp;&nbsp;Host:&nbsp;<b></font>\n");
		print("  </font>\n");
		print("  </td>\n");
	}
	print("  <td bgcolor='" . $bg_color . "'>\n");
	print("  <font face='Verdana, Tahoma, sans-serif' color='" . $fg_color . "'>\n");
	if($mode eq "localhost" || $mode eq "pc") {
		$title = $HOSTNAME;
	} elsif($mode eq "multihost") {
		my %rgraphs = reverse %GRAPHS;
		$graph = $graph eq "all" ? "_system1" : $graph;
		if(substr($graph, 0, 4) eq "_net") {
			$str = "_net" . substr($graph, 5, 1);
			$title = $rgraphs{$str};
			$title =~ s/xxx //;
		} elsif(substr($graph, 0, 5) eq "_port") {
			$str = substr($graph, 0, 5);
			$n = substr($graph, 5, 1);
			$title = $rgraphs{$str};
			$title .= " " . $PORT_LIST[$n];
			$title .= " (" . $PORT_NAME[$n] . ")";
		} else {
			$title = $rgraphs{$graph};
		}
	}
	$title =~ s/ /&nbsp;/g;
	print("    <font size='5'><b>&nbsp;&nbsp;$title&nbsp;&nbsp;</b></font>\n");
	print("  </font>\n");
	print("  </td>\n");
		print("  <td bgcolor='" . $title_bg_color . "'>\n");
		print("  <font face='Verdana, Tahoma, sans-serif' color='" . $title_fg_color . "'>\n");
		print("    <font size='5'><b>&nbsp;&nbsp;last&nbsp;$when&nbsp;&nbsp;<b></font>\n");
		print("  </font>\n");
		print("  </td>\n");
	print("  </tr>\n");
	print("  </table>\n");
	print("  <font face='Verdana, Tahoma, sans-serif' color='" . $fg_color . "'>\n");
	print("    <h4><font color='#888888'>" . strftime("%a %b %e %H:%M:%S %Z %Y", localtime) . "</font></h4>\n");
}

if($mode eq "localhost") {
	foreach my $g (@GRAPH_NAME) {
		if($GRAPH_ENABLE{$g} eq "Y") {
			if($graph eq "all" || $graph =~ /$g/) {
				if(eval {&$g($g, $GRAPH_TITLE{$g});}) {
					print("  <br>\n");
				}
			}
		}
	}
} elsif($mode eq "multihost") {
	multihost();
} elsif($mode eq "pc") {
	pc();
}

if(!$silent) {
	print("\n");
	print("  </font>\n");
	print("  </center>\n");
	print("  <p>\n");
	print("  <a href='http://www.monitorix.org'><img src='" . $URL . "logo_bot.png' border='0'></a>\n");
	print("  <br>\n");
	print("  <font face='Verdana, Tahoma, sans-serif' color='" . $fg_color . "' size='-2'>\n");
	print("Copyright &copy; 2005-2011 Jordi Sanfeliu\n");
	print("  </font>\n");
	print("  </body>\n");
	print("</html>\n");
}

exit(0);
