#!/usr/bin/perl
#
# @(#) Fibranet NSP, SL
# Copyright (C) 2005-2010 by Jordi Sanfeliu <jordi@fibranet.cat>
#

use strict;
use warnings;
use Time::Local;
use File::Copy;
use IO::Socket;
use RRDs;

our $MONITORIX_VER = "1.5.2";
our $TITLE;
our $OSTYPE;
our $SAMBAVER;
our $MULTIHOST;
our $REPORT_LANG;
our $THEME_COLOR;
our $INETIF;
our %PATH_CRON;
our %PATH_BIN;
our %BASE_WWW;
our %BASE_DIR;
our $USAGE_DIR;
our $REPORT_DIR;
our $PORT01;
our $PORT02;
our $PORT03;
our $PORT04;
our $PORT05;
our $PORT06;
our $PORT07;
our $PORT08;
our $PORT09;
our $PORT10;
our $PORT11;
our $PORT12;
our $NGINX_PORT;
our @MNT_LIST;
our @NET_LIST;
our @PC_LIST;
our @PC_IP;
our @NET_LIMIT;
our @SERV_LIST;

our %SMBSTATUS;
our %MACUSERS;
our %HTTP_LOG;
our %SECURE_LOG;
our %MAIL_LOG;

our $CPU_RRD;
our $KERN_RRD;
our $PROC_RRD;
our $DISK_RRD;
our $NET_RRD;
our $SERVU_RRD;
our $MAIL_RRD;
our $PORT_RRD;
our $USER_RRD;
our $NGINX_RRD;
our $INT_RRD;
our $PC_RRD;
our $HPTEMP_RRD;
our $LMTEMP_RRD;

our $MACHINE;
our %SENSORS_LIST;
our @HDDTEMP_LIST;
our @GRAPH_NAME;
our %GRAPH_TITLE;
our %GRAPHS;

our $LOGO;
our $TIT;
our $BLANK;
our $SIGN;
our $PC_DEFMAIL;
our @PC_MAIL;
our $IPTABLES;
our $PVS = "pvs";
our $EVMS = "evms_query";

our $TIMEOUT = 15;

our $ALERT_CPU_TIMEINTVL = 0;
our $ALERT_CPU_THRESHOLD = 0;
our $ALERT_CPU_SCRIPT = "";

our $ALERT_DISK_TIMEINTVL = 0;
our $ALERT_DISK_THRESHOLD = 100;
our $ALERT_DISK_SCRIPT = "";

our $MTA;
our $GREYLIST;
our %MILTER_GL;

our $TMP_LASTKERN = "/tmp/monitorix_lastkern";
our $TMP_LASTPROC = "/tmp/monitorix_lastproc";
our $TMP_LASTCPU15 = "/tmp/monitorix_lastcpu15";
our $TMP_LASTDISK = "/tmp/monitorix_lastdisk";
our $TMP_MAILLOG = "/tmp/monitorix_maillog";

# The current two locations where the configuration file can reside.
if(-f "/etc/monitorix.conf") {
	require "/etc/monitorix.conf";
} elsif(-f "/usr/local/etc/monitorix.conf") {
	require "/usr/local/etc/monitorix.conf";
} else {
	die("Monitorix configuration file not found!.\n");
}

our $ENABLE_CROND_MAIL = defined($ENABLE_CROND_MAIL) ? $ENABLE_CROND_MAIL : "Y";

our $ENABLE_CPU = defined($ENABLE_CPU) ? $ENABLE_CPU : "Y";
our $ENABLE_KERN = defined($ENABLE_KERN) ? $ENABLE_KERN : "Y";
our $ENABLE_PROC = defined($ENABLE_PROC) ? $ENABLE_PROC : "Y";
our $ENABLE_DISK = defined($ENABLE_DISK) ? $ENABLE_DISK : "Y";
our $ENABLE_NET = defined($ENABLE_NET) ? $ENABLE_NET : "Y";
our $ENABLE_SERVU = defined($ENABLE_SERVU) ? $ENABLE_SERVU : "Y";
our $ENABLE_MAIL = defined($ENABLE_MAIL) ? $ENABLE_MAIL : "Y";
our $ENABLE_PORT = defined($ENABLE_PORT) ? $ENABLE_PORT : "Y";
our $ENABLE_USER = defined($ENABLE_USER) ? $ENABLE_USER : "Y";
our $ENABLE_NGINX = defined($ENABLE_NGINX) ? $ENABLE_NGINX : "N";
our $ENABLE_INT = defined($ENABLE_INT) ? $ENABLE_INT : "Y";
our $ENABLE_LMTEMP = defined($ENABLE_LMTEMP) ? $ENABLE_LMTEMP : "N";
our $ENABLE_ALERTS = defined($ENABLE_ALERTS) ? $ENABLE_ALERTS : "N";

# The current two locations where the iptables command can reside.
if( -x "/usr/sbin/iptables" ) {
	$IPTABLES = "/usr/sbin/iptables";
}
if( -x "/sbin/iptables" ) {
	$IPTABLES = "/sbin/iptables";
}

if(($#ARGV+1) != 1) {
	syntax();
	exit(1);
}

if($ARGV[0] eq "init") {
	upgrade();
	init();
}
elsif($ARGV[0] eq "stop") {
	stop();
}
elsif($ARGV[0] eq "collect") {
	if($IPTABLES) {
		get_counters();
		reset_counters();
	}
}
elsif($ARGV[0] eq "create") {
	rrd_cpu() unless $ENABLE_CPU eq "N";
	rrd_kern() unless $ENABLE_KERN eq "N";
	rrd_proc() unless $ENABLE_PROC eq "N";
	if($MACHINE eq "ML310") {
		rrd_temp_ML310();
	}
	elsif($MACHINE eq "ML330") {
		rrd_temp_ML350();
	}
	elsif($MACHINE eq "ML350") {
		rrd_temp_ML350();
	}
	elsif($MACHINE eq "ML570") {
		rrd_temp_ML570();
	}
	rrd_lmtemp() unless $ENABLE_LMTEMP eq "N";
	rrd_disk() unless $ENABLE_DISK eq "N";
	rrd_net() unless $ENABLE_NET eq "N";
	rrd_servu() unless $ENABLE_SERVU eq "N";
	rrd_mail() unless $ENABLE_MAIL eq "N";
	rrd_port() unless $ENABLE_PORT eq "N";
	rrd_user() unless $ENABLE_USER eq "N";
	rrd_nginx() unless $ENABLE_NGINX eq "N";
	rrd_ints() unless $ENABLE_INT eq "N";
	if(scalar(@PC_LIST) != 0) {
		rrd_pc();
	}
}
elsif($ARGV[0] eq "report") {
	if(scalar(@PC_LIST) != 0) {
		report();
	}
}
elsif($ARGV[0] eq "update") {
	update();
}
else {
	syntax();
	exit(1);
}
exit(0);

sub syntax {
	print "Monitorix v" . $MONITORIX_VER . "\n";
	print "Copyright (C) 2005-2010 by Jordi Sanfeliu <jordi\@fibranet.cat>\n";
	print "\n";
	print "This program must be called by the Monitorix initialization script.\n";
	print "Usage: monitorix.pl [init|stop|collect|create|report|update]\n";
	print "\n";
}

# =========================================================
# 			UPGRADE OPTION
# =========================================================
#
# Some ideas of this upgrade function has taken from the scripts found in the
# contribute directory of RRDtool project.
# Written by Joost Cassee
# Copyright 2005 Bateau Knowledge
# http://dev.bateauknowledge.nl/trac/wiki/RrdAddSource
#
sub upgrade {

# CPU Database Upgrade (0.7.x -> 0.8.0)
# =====================================
# In the 0.8 version there was some changes in the process and memory graphs.
# Since then there are 2 new values to control the sleeping and running
# processes and 2 other new values to cover different memory managements in
# order to support FreeBSD and others systems.
# This function verifies all of this and upgrades (if needed) the CPU.rrd
# database.
#
	if(stat($CPU_RRD)) {
		my $LASTDS;
		my $hash;
		my $CPU_RRD_NEW = $CPU_RRD . ".new";
		my $result = 0;

		$hash = RRDs::info($CPU_RRD);
		foreach my $key (keys %$hash) {
			if(($key =~ /ds\[cpu_npslp\]/) ||
			($key =~ /ds\[cpu_nprun\]/) ||
			($key =~ /ds\[cpu_mval1\]/) ||
			($key =~ /ds\[cpu_mval2\]/)) {
				++$result;
			}
		}
		if(!($result / 7)) {
			print "\n";
			print "INFORMATION\n";
			print "-----------\n";
			print "Monitorix has detected that the CPU.rrd database needs to be upgraded to a newer version.\n";
			print "Upgrading in progress ...\n";
			if($RRDs::VERSION > 1.2) {
				$LASTDS = "UNKN";
			} else {
				$LASTDS = "U";
			}
			open(IN, "rrdtool dump $CPU_RRD|") or die("$!");
			open(OUT, "|rrdtool restore - $CPU_RRD_NEW") or die("$!");
			while(<IN>) {
				if(/<!-- Round Robin Archives -->/) {
					print OUT <<EOF;
        <ds>
                <name> cpu_npslp </name>
                <type> GAUGE </type>
                <minimal_heartbeat> 120 </minimal_heartbeat>
                <min> 0.0000000000e+00 </min>
                <max> NaN </max>

                <!-- PDP Status -->
                <last_ds> $LASTDS </last_ds>
                <value> 0.0000000000e+00 </value>
                <unknown_sec> 0 </unknown_sec>
        </ds>

        <ds>
                <name> cpu_nprun </name>
                <type> GAUGE </type>
                <minimal_heartbeat> 120 </minimal_heartbeat>
                <min> 0.0000000000e+00 </min>
                <max> NaN </max>

                <!-- PDP Status -->
                <last_ds> $LASTDS </last_ds>
                <value> 0.0000000000e+00 </value>
                <unknown_sec> 0 </unknown_sec>
        </ds>

        <ds>
                <name> cpu_mval1 </name>
                <type> GAUGE </type>
                <minimal_heartbeat> 120 </minimal_heartbeat>
                <min> 0.0000000000e+00 </min>
                <max> NaN </max>

                <!-- PDP Status -->
                <last_ds> $LASTDS </last_ds>
                <value> 0.0000000000e+00 </value>
                <unknown_sec> 0 </unknown_sec>
        </ds>

        <ds>
                <name> cpu_mval2 </name>
                <type> GAUGE </type>
                <minimal_heartbeat> 120 </minimal_heartbeat>
                <min> 0.0000000000e+00 </min>
                <max> NaN </max>

                <!-- PDP Status -->
                <last_ds> $LASTDS </last_ds>
                <value> 0.0000000000e+00 </value>
                <unknown_sec> 0 </unknown_sec>
        </ds>

EOF
				}
				if($RRDs::VERSION > 1.2) {
					if(/<\/cdp_prep>/) {
						print OUT <<EOF;
                        <ds>
                        <primary_value> 0.0000000000e+00 </primary_value>
                        <secondary_value> NaN </secondary_value>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <primary_value> 0.0000000000e+00 </primary_value>
                        <secondary_value> NaN </secondary_value>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <primary_value> 0.0000000000e+00 </primary_value>
                        <secondary_value> NaN </secondary_value>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <primary_value> 0.0000000000e+00 </primary_value>
                        <secondary_value> NaN </secondary_value>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
EOF
					}
				} else {
					if(/<\/cdp_prep>/) {
						print OUT <<EOF;
                        <ds>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
EOF
					}
				}
				s#</row>#<v> NaN </v></row>#;
				s#</row>#<v> NaN </v></row>#;
				s#</row>#<v> NaN </v></row>#;
				s#</row>#<v> NaN </v></row>#;
				print OUT $_;
			}
			close(IN);
			close(OUT);
			print "Done.\n";
			rename($CPU_RRD_NEW, $CPU_RRD);
		}
	}

# INT Database Upgrade (0.8.2 -> 0.8.3)
# =====================================
# Since the 0.8.3 version, Monitorix extended the number of supported interrupts
# from 16 (0-15) to 256 (0-255). This change is welcomed on newer systems.
#
	if(stat($INT_RRD)) {
		my $LASTDS;
		my $hash;
		my $INT_RRD_NEW = $INT_RRD . ".new";
		my $result = 0;
		my $n;

		$hash = RRDs::info($INT_RRD);
		foreach my $key (keys %$hash) {
			if($key =~ /ds\[int_/) {
				++$result;
			}
		}
		if(($result / 7) == 16 ) {
			print "\n";
			print "INFORMATION\n";
			print "-----------\n";
			print "Monitorix has detected that the INT.rrd database needs to be upgraded to a newer version.\n";
			print "Upgrading in progress ... (this may take a while)\n";
			$LASTDS = "0";
			open(IN, "rrdtool dump $INT_RRD|") or die("$!");
			open(OUT, "|rrdtool restore - $INT_RRD_NEW") or die("$!");
			while(<IN>) {
				if(/<!-- Round Robin Archives -->/) {
					for($n = 16; $n < 256; $n++) {
						print OUT <<EOF;
        <ds>
                <name> int_$n </name>
                <type> COUNTER </type>
                <minimal_heartbeat> 120 </minimal_heartbeat>
                <min> 0.0000000000e+00 </min>
                <max> NaN </max>

                <!-- PDP Status -->
                <last_ds> $LASTDS </last_ds>
                <value> 0.0000000000e+00 </value>
                <unknown_sec> 0 </unknown_sec>
        </ds>

EOF
					}
				}
				if($RRDs::VERSION > 1.2) {
					for($n = 16; $n < 256; $n++) {
						if(/<\/cdp_prep>/) {
							print OUT <<EOF;
                        <ds>
                        <primary_value> 0.0000000000e+00 </primary_value>
                        <secondary_value> NaN </secondary_value>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
EOF
						}
					}
				} else {
					for($n = 16; $n < 256; $n++) {
						if(/<\/cdp_prep>/) {
							print OUT <<EOF;
                        <ds>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
EOF
						}
					}
				}
				for($n = 16; $n < 256; $n++) {
					s#</row>#<v> NaN </v></row>#;
				}
				print OUT $_;
			}
			close(IN);
			close(OUT);
			print "Done.\n";
			rename($INT_RRD_NEW, $INT_RRD);
		}
	}

# NET Database Upgrade (0.8.2 -> 0.8.3)
# =====================================
# Since the 0.8.3 version, Monitorix extended the number of supported NICs
# from 3 to 5.
#
	if(stat($NET_RRD)) {
		my $LASTDS;
		my $hash;
		my $NET_RRD_NEW = $NET_RRD . ".new";
		my $result = 0;
		my $n;

		$hash = RRDs::info($NET_RRD);
		foreach my $key (keys %$hash) {
			if($key =~ /ds\[net/) {
				++$result;
			}
		}
		if(($result / 7 / 6) == 3 ) {
			print "\n";
			print "INFORMATION\n";
			print "-----------\n";
			print "Monitorix has detected that the NET.rrd database needs to be upgraded to a newer version.\n";
			print "Upgrading in progress ...\n";
			$LASTDS = "0";
			open(IN, "rrdtool dump $NET_RRD|") or die("$!");
			open(OUT, "|rrdtool restore - $NET_RRD_NEW") or die("$!");
			while(<IN>) {
				if(/<!-- Round Robin Archives -->/) {
					for($n = 3; $n < 6; $n++) {
						print OUT <<EOF;
        <ds>
                <name> net${n}_bytes_in </name>
                <type> COUNTER </type>
                <minimal_heartbeat> 120 </minimal_heartbeat>
                <min> 0.0000000000e+00 </min>
                <max> NaN </max>

                <!-- PDP Status -->
                <last_ds> $LASTDS </last_ds>
                <value> 0.0000000000e+00 </value>
                <unknown_sec> 0 </unknown_sec>
        </ds>

        <ds>
                <name> net${n}_bytes_out </name>
                <type> COUNTER </type>
                <minimal_heartbeat> 120 </minimal_heartbeat>
                <min> 0.0000000000e+00 </min>
                <max> NaN </max>

                <!-- PDP Status -->
                <last_ds> $LASTDS </last_ds>
                <value> 0.0000000000e+00 </value>
                <unknown_sec> 0 </unknown_sec>
        </ds>

        <ds>
                <name> net${n}_packs_in </name>
                <type> COUNTER </type>
                <minimal_heartbeat> 120 </minimal_heartbeat>
                <min> 0.0000000000e+00 </min>
                <max> NaN </max>

                <!-- PDP Status -->
                <last_ds> $LASTDS </last_ds>
                <value> 0.0000000000e+00 </value>
                <unknown_sec> 0 </unknown_sec>
        </ds>

        <ds>
                <name> net${n}_packs_out </name>
                <type> COUNTER </type>
                <minimal_heartbeat> 120 </minimal_heartbeat>
                <min> 0.0000000000e+00 </min>
                <max> NaN </max>

                <!-- PDP Status -->
                <last_ds> $LASTDS </last_ds>
                <value> 0.0000000000e+00 </value>
                <unknown_sec> 0 </unknown_sec>
        </ds>

        <ds>
                <name> net${n}_error_in </name>
                <type> COUNTER </type>
                <minimal_heartbeat> 120 </minimal_heartbeat>
                <min> 0.0000000000e+00 </min>
                <max> NaN </max>

                <!-- PDP Status -->
                <last_ds> $LASTDS </last_ds>
                <value> 0.0000000000e+00 </value>
                <unknown_sec> 0 </unknown_sec>
        </ds>

        <ds>
                <name> net${n}_error_out </name>
                <type> COUNTER </type>
                <minimal_heartbeat> 120 </minimal_heartbeat>
                <min> 0.0000000000e+00 </min>
                <max> NaN </max>

                <!-- PDP Status -->
                <last_ds> $LASTDS </last_ds>
                <value> 0.0000000000e+00 </value>
                <unknown_sec> 0 </unknown_sec>
        </ds>

EOF
					}
				}
				if($RRDs::VERSION > 1.2) {
					for($n = 3; $n < 6; $n++) {
						if(/<\/cdp_prep>/) {
							print OUT <<EOF;
                        <ds>
                        <primary_value> 0.0000000000e+00 </primary_value>
                        <secondary_value> NaN </secondary_value>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <primary_value> 0.0000000000e+00 </primary_value>
                        <secondary_value> NaN </secondary_value>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <primary_value> 0.0000000000e+00 </primary_value>
                        <secondary_value> NaN </secondary_value>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <primary_value> 0.0000000000e+00 </primary_value>
                        <secondary_value> NaN </secondary_value>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <primary_value> 0.0000000000e+00 </primary_value>
                        <secondary_value> NaN </secondary_value>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <primary_value> 0.0000000000e+00 </primary_value>
                        <secondary_value> NaN </secondary_value>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
EOF
						}
					}
				} else {
					for($n = 3; $n < 6; $n++) {
						if(/<\/cdp_prep>/) {
							print OUT <<EOF;
                        <ds>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
                        <ds>
                        <value> NaN </value>
                        <unknown_datapoints> 0 </unknown_datapoints>
                        </ds>
EOF
						}
					}
				}
				for($n = 3; $n < 6; $n++) {
					s#</row>#<v> NaN </v></row>#;
					s#</row>#<v> NaN </v></row>#;
					s#</row>#<v> NaN </v></row>#;
					s#</row>#<v> NaN </v></row>#;
					s#</row>#<v> NaN </v></row>#;
					s#</row>#<v> NaN </v></row>#;
				}
				print OUT $_;
			}
			close(IN);
			close(OUT);
			print "Done.\n";
			rename($NET_RRD_NEW, $NET_RRD);
		}
	}

# TEMP Database Change (0.9.2 -> 1.0.0)
# =====================================
# Since the 1.0.0 version, Monitorix added a new temperature graph based on
# "lm-sensors" and "hddtemp" tools. The HP temperatures graph is now under the
# renamed file hptemp.rrd and the new lm-sensors/hddtemp based temperatures
# will use the lmtemp.rrd file.
#
	if(!$HPTEMP_RRD || !$LMTEMP_RRD) {
		print "FATAL: Please upgrade your /etc/monitorix.conf file to include the definition of the \$LMTEMP_RRD option and the new ones appeared since the 1.0.0 version. See the new configuration file for the details.\n";
		exit(1);
	}
	my $TEMP_RRD = $BASE_DIR{$OSTYPE} . "temp.rrd";
	if(stat($TEMP_RRD)) {
		rename($TEMP_RRD, $HPTEMP_RRD);
		print "\n";
		print "INFORMATION\n";
		print "-----------\n";
		print "Monitorix has renamed your temperatures database from '$TEMP_RRD' to '$HPTEMP_RRD'.\n";
		print "Done.\n";
	}
}


# =========================================================
# 			INIT OPTION
# =========================================================

sub init {
	my $n;
	my $O_CRON = $PATH_CRON{$OSTYPE};
	my $O_INDEX = $BASE_WWW{$OSTYPE} . "/monitorix/index.html.tmp";
	my $crontabuser;
	my $ncpu;
	my $eths;
	my $mailto;
	my $bgcolor;
	my $table_back_color;
	my $title_back_color;
	my $title_fore_color;

	# Check for global consistencies
	if($ENABLE_ALERTS eq "Y") {
		(-x $ALERT_CPU_SCRIPT) or die("\nERROR: '" . $ALERT_CPU_SCRIPT . "' is an invalid script.");
	}

	unlink($TMP_LASTKERN, $TMP_LASTPROC, $TMP_LASTCPU15, $TMP_MAILLOG, $TMP_LASTDISK);

	$ncpu = 0;
	if(($OSTYPE eq "Linux-RHFC") ||
	  ($OSTYPE eq "Linux-Arch") ||
	  ($OSTYPE eq "Linux-Debian") ||
	  ($OSTYPE eq "Linux-Gentoo") ||
	  ($OSTYPE eq "Linux-Slack") ||
	  ($OSTYPE eq "Linux-SuSE") ||
	  ($OSTYPE eq "Linux-Generic")) {
		$crontabuser = $OSTYPE eq "Linux-Arch" ? "" : "root";

# We call here "pvs -a" command just to make sure that it's completely removed
# a strange message about "Medium not found" in fresh CentOS 5 installations
# with LVM.
		system("$PVS -a >/dev/null 2>&1");

		$ncpu = `grep -w processor /proc/cpuinfo | tail -1 | awk '{ print \$3 }'`;
		chomp($ncpu);
		$ncpu++;
	} elsif($OSTYPE eq "FreeBSD") {
		$ncpu = `/sbin/sysctl -n hw.ncpu`;
		chomp($ncpu);
		$crontabuser = "";
	}

#* * * * * root ps -C monitorix.pl --no-headers >/dev/null || /usr/sbin/monitorix.pl update
	$mailto = "";
	if($ENABLE_CROND_MAIL ne "Y") {
		$mailto = "MAILTO=\"\"";
	}
	open(OCRON, "> $O_CRON");
	print OCRON <<EOF;
# @(#) Fibranet NSP, SL
# Copyright (C) 2005-2010 by Jordi Sanfeliu <jordi\@fibranet.cat>
#
PATH=/sbin:/bin:/usr/sbin:/usr/bin
$mailto

* * * * * $crontabuser $PATH_BIN{$OSTYPE} update
00 00 * * * $crontabuser $PATH_BIN{$OSTYPE} collect
01 00 1 * * $crontabuser $PATH_BIN{$OSTYPE} report

EOF
	close(OCRON);

	if($THEME_COLOR eq "black") {
		$bgcolor = "#000000";
		$table_back_color = "#888888";
		$title_back_color = "#333333";
		$title_fore_color = "#888800";

	} else {
		$bgcolor = "#ffffff";
		$table_back_color = "#CCCCCC";
		$title_back_color = "#777777";
		$title_fore_color = "#CCCC00";
	}

	open(OHTML, "> $O_INDEX");
	print OHTML <<EOF;
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2 Final//EN">
<html>
 <head>
  <title>$TITLE</title>
  <link rel="shortcut icon" href="/monitorix/monitorixico.png">
 </head>
 <body bgcolor="$bgcolor" text="#888888" vlink="#888888" link="#888888">
  <center>
  <p>
  <br>
  <font face="Verdana, Tahoma, sans-serif">
  <table>
    <tr>
      <td>
        <a href="http://www.monitorix.org/"><img src="logo_top.png" border=0></a>
      </td>
    </tr>
    <tr>
      <td>
        <h2 align="right">v$MONITORIX_VER</h2>
      </td>
    </tr>
  </table>
  <p>
  <form action="/cgi-bin/monitorix.cgi" method="get">
    <table cellspacing=5 cellpadding=0 bgcolor="$table_back_color" border=1>
      <tr>
        <td bgcolor="$title_back_color">
          <font color="$title_fore_color">
<b>&nbsp;Hostname&nbsp;</b>
          </font>
       </td>
        <td bgcolor="$title_back_color">
          <font color="$title_fore_color">
<b>&nbsp;Graph&nbsp;</b>
          </font>
       </td>
     </tr>
      <tr>
        <td bgcolor="$bgcolor">
<!MARK 1>
        </td>
        <td bgcolor="$bgcolor">
<!MARK 2>
        </td>
     </tr>
    </table>
    <p>
    <table cellspacing=5 cellpadding=0 bgcolor="$table_back_color" border=1>
     <tr>
        <td bgcolor="$title_back_color">
          <input type="radio" checked name="when" value="day">
            <font color="$title_fore_color"><b>Daily&nbsp;</b></font>
        </td>
        <td bgcolor="$title_back_color">
          <input type="radio" name="when" value="week">
            <font color="$title_fore_color"><b>Weekly&nbsp;</b></font>
        </td>
        <td bgcolor="$title_back_color">
          <input type="radio" name="when" value="month">
            <font color="$title_fore_color"><b>Monthly&nbsp;</b></font>
        </td>
        <td bgcolor="$title_back_color">
          <input type="radio" name="when" value="year">
            <font color="$title_fore_color"><b>Yearly&nbsp;</b></font>
        </td>
      </tr>
    </table>
    <p>
    <input type="hidden" name="color" value="$THEME_COLOR">
    <input type="submit" value="    Ok    ">
  </form>
  </font>
  </center>
 </body>
</html>
EOF
	close(OHTML);
	my $I_INDEX=$BASE_WWW{$OSTYPE} . "/monitorix/index.html.tmp";
	$O_INDEX=$BASE_WWW{$OSTYPE} . "/monitorix/index.html";

	$eths = 0;
	$eths = scalar(@NET_LIST);

	open(IHTML, "< $I_INDEX");
	open(OHTML, "> $O_INDEX");
	while(<IHTML>) {
		if(/<!MARK 1/) {
#			s/<OPTION>.*$/\$ENV{HOSTNAME}<\/OPTION>/;
			print OHTML "          <select name='mode' size=1>\n";
			print OHTML "            <option value='localhost'>Local Host<\/option>\n";
			if((scalar(@SERV_LIST)) && ($MULTIHOST eq "Y")) {
				print OHTML "            <optgroup label='Multihost'>\n";
				print OHTML "            <option value='multihost.all'>All hosts<\/option>\n";
				for($n = 0; $n < scalar(@SERV_LIST); $n += 2) {
					print OHTML "              <option value='multihost." . ($n / 2). "'>" . $SERV_LIST[$n] . "<\/option>\n";
                                }
				print OHTML "            <\/optgroup>\n";
			}
			if(scalar(@PC_LIST)) {
				print OHTML "            <optgroup label='LAN PCs'>\n";
				print OHTML "              <option value='pc.all'>All LAN PCs<\/option>\n";
				for($n = 0; $n < scalar(@PC_LIST); $n++) {
					print OHTML "              <option value='pc." . $n . "'>" . $PC_LIST[$n] . "<\/option>\n";
                                }
				print OHTML "            <\/optgroup>\n";
			}
			print OHTML "          <\/select>\n";
			next;
		}
		if(/<!MARK 2/) {
			my $gname;
			my $n2;
			my %rgraphs = reverse %GRAPHS;

			print OHTML "          <select name='graph' size=1>\n";
			print OHTML "            <option value='all'>All graphs<\/option>\n";
			for($n = 0; $n < scalar(@GRAPH_NAME); $n++) {
				if(($GRAPH_NAME[$n] eq "cpu" && $ENABLE_CPU eq "Y") ||
				($GRAPH_NAME[$n] eq "kern" && $ENABLE_KERN eq "Y") ||
				($GRAPH_NAME[$n] eq "proc" && $ncpu > 1 && $ENABLE_PROC eq "Y") ||
				($GRAPH_NAME[$n] eq "hptemp" && $MACHINE) ||
				($GRAPH_NAME[$n] eq "lmtemp" && $ENABLE_LMTEMP eq "Y") ||
				($GRAPH_NAME[$n] eq "disk" && $ENABLE_DISK eq "Y") ||
				($GRAPH_NAME[$n] eq "net" && $ENABLE_NET eq "Y") ||
				($GRAPH_NAME[$n] eq "servu" && $ENABLE_SERVU eq "Y") ||
				($GRAPH_NAME[$n] eq "mail" && $ENABLE_MAIL eq "Y") ||
				($GRAPH_NAME[$n] eq "port" && $ENABLE_PORT eq "Y") ||
				($GRAPH_NAME[$n] eq "user" && $ENABLE_USER eq "Y") ||
				($GRAPH_NAME[$n] eq "nginx" && $ENABLE_NGINX eq "Y") ||
				($GRAPH_NAME[$n] eq "int" && $ENABLE_INT eq "Y")) {
				print OHTML "            <optgroup label='" . $GRAPH_TITLE{$GRAPH_NAME[$n]} . "'>\n";
				for($n2 = 0; $n2 <= 53; $n2++) {
					if($GRAPH_NAME[$n] eq "net" && $n2 > ($eths * 10)) {
						next;
					}
					if($GRAPH_NAME[$n] eq "port" ||
					$GRAPH_NAME[$n] eq "net" ||
					($GRAPH_NAME[$n] eq "proc" && $n2 < $ncpu)) {
						$gname = sprintf("_%s%02d", $GRAPH_NAME[$n], $n2);
					} else {
						$gname = "_" . $GRAPH_NAME[$n] . $n2;
					}
					if($rgraphs{$gname}) {
						print OHTML "              <option value='" . $gname ."'>" . $rgraphs{$gname} . "<\/option>\n";
					}
				}
				}
				print OHTML "            <\/optgroup>\n";
			}
			print OHTML "          <\/select>\n";
			next;
		}
		print OHTML $_;
	}
	close(IHTML);
	close(OHTML);
	unlink($I_INDEX);

	if($IPTABLES && $ENABLE_PORT eq "Y" && $OSTYPE ne "FreeBSD") {
		# create port counters
		if($PORT01) {
			system("$IPTABLES -N PORT01_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT01_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT01 -j PORT01_IN -c 0 0");
			system("$IPTABLES -N PORT01_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT01_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT01 -j PORT01_OUT -c 0 0");
		}
		if($PORT02) {
			system("$IPTABLES -N PORT02_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT02_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT02 -j PORT02_IN -c 0 0");
			system("$IPTABLES -N PORT02_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT02_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT02 -j PORT02_OUT -c 0 0");
		}
		if($PORT03) {
			system("$IPTABLES -N PORT03_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT03_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT03 -j PORT03_IN -c 0 0");
			system("$IPTABLES -N PORT03_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT03_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT03 -j PORT03_OUT -c 0 0");
		}
		if($PORT04) {
			system("$IPTABLES -N PORT04_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT04_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT04 -j PORT04_IN -c 0 0");
			system("$IPTABLES -N PORT04_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT04_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT04 -j PORT04_OUT -c 0 0");
		}			
		if($PORT05) {
			system("$IPTABLES -N PORT05_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT05_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT05 -j PORT05_IN -c 0 0");
			system("$IPTABLES -N PORT05_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT05_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT05 -j PORT05_OUT -c 0 0");
		}
		if($PORT06) {
			system("$IPTABLES -N PORT06_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT06_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT06 -j PORT06_IN -c 0 0");
			system("$IPTABLES -N PORT06_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT06_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT06 -j PORT06_OUT -c 0 0");
		}
		if($PORT07) {
			system("$IPTABLES -N PORT07_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT07_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT07 -j PORT07_IN -c 0 0");
			system("$IPTABLES -N PORT07_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT07_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT07 -j PORT07_OUT -c 0 0");
		}
		if($PORT08) {
			system("$IPTABLES -N PORT08_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT08_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT08 -j PORT08_IN -c 0 0");
			system("$IPTABLES -N PORT08_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT08_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT08 -j PORT08_OUT -c 0 0");
		}
		if($PORT09) {
			system("$IPTABLES -N PORT09_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT09_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT09 -j PORT09_IN -c 0 0");
			system("$IPTABLES -N PORT09_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT09_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT09 -j PORT09_OUT -c 0 0");
		}
		if($PORT10) {
			system("$IPTABLES -N PORT10_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT10_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT10 -j PORT10_IN -c 0 0") unless !$PORT10;
			system("$IPTABLES -N PORT10_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT10_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT10 -j PORT10_OUT -c 0 0");
		}
		if($PORT11) {
			system("$IPTABLES -N PORT11_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT11_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT11 -j PORT11_IN -c 0 0");
			system("$IPTABLES -N PORT11_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT11_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT11 -j PORT11_OUT -c 0 0");
		}
		if($PORT12) {
			system("$IPTABLES -N PORT12_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q PORT12_IN) || $IPTABLES -I INPUT -p tcp --dport $PORT12 -j PORT12_IN -c 0 0");
			system("$IPTABLES -N PORT12_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q PORT12_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $PORT12 -j PORT12_OUT -c 0 0");
		}
		if($ENABLE_NGINX eq "Y" && defined($NGINX_PORT)) {
			system("$IPTABLES -N NGINX_IN 2>/dev/null");
			system("($IPTABLES -L INPUT | grep -q NGINX_IN) || $IPTABLES -I INPUT -p tcp --dport $NGINX_PORT -j NGINX_IN -c 0 0");
			system("$IPTABLES -N NGINX_OUT 2>/dev/null");
			system("($IPTABLES -L OUTPUT | grep -q NGINX_OUT) || $IPTABLES -I OUTPUT -p tcp --sport $NGINX_PORT -j NGINX_OUT -c 0 0");
		}
	}

	if($IPTABLES && $OSTYPE ne "FreeBSD") {
		# create PC counters
		if(scalar(@PC_LIST)) {
			my $ip;
			for($n = 0; $n < scalar(@PC_LIST); $n++) {
				if(!($ip = $PC_IP[$n])) {
					if(!(gethostbyname($PC_LIST[$n]))) {
						print "DNS problem with: ", $PC_LIST[$n], "\n";
					}
					$ip = inet_ntoa((gethostbyname($PC_LIST[$n]))[4]);
					$ip = $ip . "/32";
				}
				system("$IPTABLES -N $PC_LIST[$n]_total 2>/dev/null");
				if(!($? >> 8)) {
					system("$IPTABLES -N $PC_LIST[$n]_daily");
					system("$IPTABLES -I FORWARD -j $PC_LIST[$n]_total");
					system("$IPTABLES -A $PC_LIST[$n]_total -s $ip -d 0/0 -o $INETIF");
					system("$IPTABLES -A $PC_LIST[$n]_total -s 0/0 -d $ip -i $INETIF");
					system("$IPTABLES -I FORWARD -j $PC_LIST[$n]_daily");
					system("$IPTABLES -A $PC_LIST[$n]_daily -s $ip -d 0/0 -o $INETIF");
					system("$IPTABLES -A $PC_LIST[$n]_daily -s 0/0 -d $ip -i $INETIF");
				}
			}
		}
	}

	# On FreeBSD systems the crontab file needs to be installed using the
	# command crontab.
	if($OSTYPE eq "FreeBSD") {
		system("crontab -u root $PATH_CRON{$OSTYPE}");
	}
}

# =========================================================
# 			STOP OPTION
# =========================================================

sub stop {
	# We need to remove the crontab file to stop Monitorix.
	if(($OSTYPE eq "Linux-RHFC") ||
	  ($OSTYPE eq "Linux-Arch") ||
	  ($OSTYPE eq "Linux-Debian") ||
	  ($OSTYPE eq "Linux-Gentoo") ||
	  ($OSTYPE eq "Linux-Slack") ||
	  ($OSTYPE eq "Linux-SuSE") ||
	  ($OSTYPE eq "Linux-Generic")) {
		unlink($PATH_CRON{$OSTYPE});
		unlink($PATH_CRON{$OSTYPE} . ".sh");	# old name
	} elsif($OSTYPE eq "FreeBSD") {
		system("echo 'y' | crontab -u root -r");
	}
}

# =========================================================
# 			COLLECT OPTION
# =========================================================

sub get_counters {
	my $n;
	my $ip;
	my $IN;
	my $OUT;
	my ($day) = (localtime(time))[3];

	for($n = 0; $n < scalar(@PC_LIST); $n++) {
		if(!($ip = $PC_IP[$n])) {
			if(!(gethostbyname($PC_LIST[$n]))) {
				print "DNS problem with: ", $PC_LIST[$n], "\n";
			}
			$ip = inet_ntoa((gethostbyname($PC_LIST[$n]))[4]);
		}
		$ip=~ s/\/\d+//;
		$IN = `$IPTABLES -nxvL $PC_LIST[$n]_daily | grep $ip | tail -1 | awk -F " " '{ print \$2 }'`;
		$OUT = `$IPTABLES -nxvL $PC_LIST[$n]_daily | grep $ip | head -1 | awk -F " " '{ print \$2 }'`;
		chomp($IN, $OUT);
		$IN = $IN ? $IN : 0;
		$OUT = $OUT ? $OUT : 0;
		{
			my $STATS=$BASE_DIR{$OSTYPE} . $USAGE_DIR . $PC_LIST[$n];
			open(PCSTATS, ">> $STATS") or die("$!");
			print PCSTATS $day, " ", $IN, " ", $OUT, "\n";
			close(PCSTATS);
		}
	}
}

sub reset_counters {
	my $n;

	for($n = 0; $n < scalar(@PC_LIST); $n++) {
		system("$IPTABLES -Z $PC_LIST[$n]_daily");
	}
}

# ==================================================
# 			REPORT OPTION
# ==================================================

sub report {
	my $n;
	my $TOT_IN;
	my $TOT_OUT;
	my $TOTAL;
	my @f_totin;
	my @f_totout;
	my @f_total;
	my $rline;
	my $month = (localtime(time))[4];	# (0-11  where 0=january)
	my $year = (localtime(time))[5] + 1900;
	my $inetpos;	
	my @graph_colors;
	my @VERSION12;
	our %BLACK;

	for($n = 0; $n < scalar(@NET_LIST); $n++) {
		if($NET_LIST[$n] eq $INETIF) {
			$inetpos = $n;
		}
	}

# Catalan month localization
	my @monthlist = ("Desembre",
	   		"Gener",
	   		"Febrer",
	   		"Mar",
	   		"Abril",
	   		"Maig",
	   		"Juny",
	   		"Juliol",
	   		"Agost",
	   		"Setembre",
	   		"Octubre",
	   		"Novembre");

	if($RRDs::VERSION > 1.2) {
		$VERSION12[0] = "--font=LEGEND:7:";
		$VERSION12[1] = "--font=TITLE:9:";
		$VERSION12[2] = "--slope-mode";
	} else {
		undef(@VERSION12);
	}

	if($THEME_COLOR eq "black") {
		$graph_colors[0] = "--color=CANVAS" . $BLACK{canvas};
		$graph_colors[1] = "--color=BACK" . $BLACK{back};
		$graph_colors[2] = "--color=FONT" . $BLACK{font};
		$graph_colors[3] = "--color=MGRID" . $BLACK{mgrid};
		$graph_colors[4] = "--color=GRID" . $BLACK{grid};
		$graph_colors[5] = "--color=FRAME" . $BLACK{frame};
		$graph_colors[6] = "--color=ARROW" . $BLACK{arrow};
		$graph_colors[7] = "--color=SHADEA" . $BLACK{shadea};
		$graph_colors[8] = "--color=SHADEB" . $BLACK{shadeb};
	}


	sub adjust($);

	for($n = 0; $n < scalar(@PC_LIST); $n++) {
		my $CURR_STATS = $BASE_DIR{$OSTYPE} . $USAGE_DIR . $PC_LIST[$n];
		my $IHTML_FILE = $BASE_DIR{$OSTYPE} . $REPORT_DIR . $REPORT_LANG . "/" . "traffic_report.html";
		my $OHTML_FILE = $BASE_DIR{$OSTYPE} . $REPORT_DIR . $REPORT_LANG . "/" . $PC_LIST[$n] . ".html";
		my $GRAPH = $BASE_DIR{$OSTYPE} . $REPORT_DIR . $REPORT_LANG . "/" . $PC_LIST[$n] . ".png";
		my @DEF;
		my $TO_ADDR;

	if (-e $CURR_STATS) {
		$TOT_IN = 0;
		$TOT_OUT = 0;
		$rline = "";
		open(PC_CURRSTATS, "< $CURR_STATS");
		$rline .= "DAY      INPUT            OUTPUT                 TOTAL\n";
		$rline .= "------------------------------------------------------------\n";
		while(<PC_CURRSTATS>) {
			my @stats = split(/ /, $_);
			chomp(@stats);
			$TOT_IN += $stats[1];
			$TOT_OUT += $stats[2];
			$TOTAL = $stats[1] + $stats[2];
			@f_totin = adjust($stats[1]);
			@f_totout = adjust($stats[2]);
			@f_total = adjust($TOTAL);
			$rline .= sprintf("%-3u %10u %3u%s %11u %3u%s %15u %3u%s\n", $stats[0], $stats[1], $f_totin[0], $f_totin[1], $stats[2], $f_totout[0], $f_totout[1], $TOTAL, $f_total[0], $f_total[1]);
		}
		close(PC_CURRSTATS);
		$rline .= "   ---------------------------------------------------------\n";
		$TOTAL = $TOT_IN + $TOT_OUT;
		@f_totin = adjust($TOT_IN);
		@f_totout = adjust($TOT_OUT);
		@f_total = adjust($TOTAL);
		$rline .= sprintf("%14.0f %3u%s %11.0f %3u%s %15.0f %3u%s\n", $TOT_IN, $f_totin[0], $f_totin[1], $TOT_OUT, $f_totout[0], $f_totout[1], $TOTAL, $f_total[0], $f_total[1]);

		open(IHTML, "< $IHTML_FILE");
		open(OHTML, "> $OHTML_FILE");
		while(<IHTML>) {
			if(/TRAFFICTRAFFICTRAFFIC/) {
				print OHTML $rline;
			} else {
				print OHTML $_;
			}
		}
		close(IHTML);
		close(OHTML);
		$DEF[0] = "DEF:B_in=$PC_RRD:pc" . ($n+1) . "_bytes_in:AVERAGE";
		$DEF[1] = "DEF:B_out=$PC_RRD:pc" . ($n+1) . "_bytes_out:AVERAGE";
		RRDs::graph("$GRAPH",
			"--title=$PC_LIST[$n] traffic  (1month)",
			"--start=-1month",
			"--imgformat=PNG",
			"--vertical-label=bytes/sec",
			"--width=450",
			"--height=150",
			"--upper-limit=$NET_LIMIT[$inetpos]",
			"--lower-limit=0",
			"--rigid",
			@VERSION12,
			@graph_colors,
			@DEF,
			"CDEF:K_in=B_in,1024,/",
			"CDEF:K_out=B_out,1024,/",
			"AREA:B_out#4444EE:Output",
			"AREA:B_in#44EE44:Input",
				"LINE1:B_out#0000EE",
				"LINE1:B_in#00EE00",
			"COMMENT:\\n",
			"COMMENT:\\n",
			"GPRINT:K_out:LAST:KB/s Output      Current\\: %5.0lf",
			"GPRINT:K_out:AVERAGE:    Average\\: %5.0lf",
			"GPRINT:K_out:MIN:    Min\\: %5.0lf",
			"GPRINT:K_out:MAX:    Max\\: %5.0lf\\n",
			"GPRINT:K_in:LAST:KB/s Input       Current\\: %5.0lf",
			"GPRINT:K_in:AVERAGE:    Average\\: %5.0lf",
			"GPRINT:K_in:MIN:    Min\\: %5.0lf",
			"GPRINT:K_in:MAX:    Max\\: %5.0lf\\n",
			"COMMENT:\\n");
		my $err = RRDs::error;
		die("ERROR: while creating $GRAPH: $err\n") if $err;
		$TO_ADDR = $PC_MAIL[$n] ? $PC_MAIL[$n] : $PC_DEFMAIL;
		if($TO_ADDR) {
			system("$BASE_DIR{$OSTYPE}$REPORT_DIR$REPORT_LANG/traffic_report.sh $TO_ADDR $monthlist[$month] $PC_LIST[$n] $OHTML_FILE $GRAPH $LOGO $TIT $BLANK $SIGN");
		}
		unlink($OHTML_FILE, $GRAPH);
		my $NEW_CURR_STATS = sprintf("%s.%02u-%u", $CURR_STATS, ($month + 1), $year);
		rename($CURR_STATS, $NEW_CURR_STATS);
	}
	}
}

sub adjust($) {
	my $bytes = (shift);
	my $adjust = 0;
	my $b = "  ";

	if($bytes > 0  &&  $bytes < 1048576) {
		$adjust = $bytes/1024;
		$b = "KB";
	}
	if($bytes > 1048576  &&  $bytes < 1073741824) {
		$adjust = $bytes/1024/1024;
		$b = "MB";
	}
	if($bytes > 1073741824  &&  $bytes < 1000000000000) {
		$adjust = $bytes/1024/1024/1024;
		$b = "GB";
	}
	return $adjust, $b;
}

# ==================================================
# 			CREATE OPTION
# ==================================================

sub rrd_cpu {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $CPU_RRD)) {
		RRDs::create($CPU_RRD,
			"--step=60",
			"DS:cpu_load1:GAUGE:120:0:U",
			"DS:cpu_load5:GAUGE:120:0:U",
			"DS:cpu_load15:GAUGE:120:0:U",
			"DS:cpu_nproc:GAUGE:120:0:U",
			"DS:cpu_mbuff:GAUGE:120:0:U",
			"DS:cpu_mcach:GAUGE:120:0:U",
			"DS:cpu_mfree:GAUGE:120:0:U",
			"DS:cpu_npslp:GAUGE:120:0:U",
			"DS:cpu_nprun:GAUGE:120:0:U",
			"DS:cpu_mval1:GAUGE:120:0:U",
			"DS:cpu_mval2:GAUGE:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $CPU_RRD: $err\n") if $err;
	}
}

sub rrd_kern {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $KERN_RRD)) {
		RRDs::create($KERN_RRD,
			"--step=60",
			"DS:kern_usr:GAUGE:120:0:U",
			"DS:kern_nic:GAUGE:120:0:U",
			"DS:kern_sys:GAUGE:120:0:U",
			"DS:kern_iow:GAUGE:120:0:U",
			"DS:kern_cs:COUNTER:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $KERN_RRD: $err\n") if $err;
	}
}

sub rrd_proc {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $PROC_RRD)) {
		RRDs::create($PROC_RRD,
			"--step=60",
			"DS:p0_usr:GAUGE:120:0:U",
			"DS:p0_nic:GAUGE:120:0:U",
			"DS:p0_sys:GAUGE:120:0:U",
			"DS:p0_iow:GAUGE:120:0:U",
			"DS:p1_usr:GAUGE:120:0:U",
			"DS:p1_nic:GAUGE:120:0:U",
			"DS:p1_sys:GAUGE:120:0:U",
			"DS:p1_iow:GAUGE:120:0:U",
			"DS:p2_usr:GAUGE:120:0:U",
			"DS:p2_nic:GAUGE:120:0:U",
			"DS:p2_sys:GAUGE:120:0:U",
			"DS:p2_iow:GAUGE:120:0:U",
			"DS:p3_usr:GAUGE:120:0:U",
			"DS:p3_nic:GAUGE:120:0:U",
			"DS:p3_sys:GAUGE:120:0:U",
			"DS:p3_iow:GAUGE:120:0:U",
			"DS:p4_usr:GAUGE:120:0:U",
			"DS:p4_nic:GAUGE:120:0:U",
			"DS:p4_sys:GAUGE:120:0:U",
			"DS:p4_iow:GAUGE:120:0:U",
			"DS:p5_usr:GAUGE:120:0:U",
			"DS:p5_nic:GAUGE:120:0:U",
			"DS:p5_sys:GAUGE:120:0:U",
			"DS:p5_iow:GAUGE:120:0:U",
			"DS:p6_usr:GAUGE:120:0:U",
			"DS:p6_nic:GAUGE:120:0:U",
			"DS:p6_sys:GAUGE:120:0:U",
			"DS:p6_iow:GAUGE:120:0:U",
			"DS:p7_usr:GAUGE:120:0:U",
			"DS:p7_nic:GAUGE:120:0:U",
			"DS:p7_sys:GAUGE:120:0:U",
			"DS:p7_iow:GAUGE:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $PROC_RRD: $err\n") if $err;
	}
}

sub rrd_lmtemp {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $LMTEMP_RRD)) {
		RRDs::create($LMTEMP_RRD,
			"--step=60",
			"DS:mb:GAUGE:120:0:U",
			"DS:cpu1:GAUGE:120:0:U",
			"DS:cpu2:GAUGE:120:0:U",
			"DS:cpu3:GAUGE:120:0:U",
			"DS:cpu4:GAUGE:120:0:U",
			"DS:fan1:GAUGE:120:0:U",
			"DS:fan2:GAUGE:120:0:U",
			"DS:fan3:GAUGE:120:0:U",
			"DS:hd1:GAUGE:120:0:U",
			"DS:hd2:GAUGE:120:0:U",
			"DS:hd3:GAUGE:120:0:U",
			"DS:hd4:GAUGE:120:0:U",
			"DS:hd5:GAUGE:120:0:U",
			"DS:hd6:GAUGE:120:0:U",
			"DS:hd7:GAUGE:120:0:U",
			"DS:hd8:GAUGE:120:0:U",
			"DS:hd9:GAUGE:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $LMTEMP_RRD: $err\n") if $err;
	}
}

sub rrd_disk {
# RRA: average data of 300*288 secs = 1 day
# RRA: average data of 300*6*336 secs = 1 week
# RRA: average data of 300*12*744 secs = 1 month
# RRA: average data of 300*288*365 secs = 1 year
# RRA: min data of 300*288 secs = 1 day
# RRA: min data of 300*6*336 secs = 1 week
# RRA: min data of 300*12*744 secs = 1 month
# RRA: min data of 300*288*365 secs = 1 year
# RRA: max data of 300*288 secs = 1 day
# RRA: max data of 300*6*336 secs = 1 week
# RRA: max data of 300*12*744 secs = 1 month
# RRA: max data of 300*288*365 secs = 1 year
# RRA: last data of 300*288 secs = 1 day
# RRA: last data of 300*6*336 secs = 1 week
# RRA: last data of 300*12*744 secs = 1 month
# RRA: last data of 300*288*365 secs = 1 year
	if(!(-e $DISK_RRD)) {
		RRDs::create($DISK_RRD,
			"--step=300",
			"DS:root_tot:GAUGE:600:0:U",
			"DS:root_used:GAUGE:600:0:U",
			"DS:root_free:GAUGE:600:0:U",
			"DS:swap_tot:GAUGE:600:0:U",
			"DS:swap_used:GAUGE:600:0:U",
			"DS:swap_free:GAUGE:600:0:U",
			"DS:mnt1_tot:GAUGE:600:0:U",
			"DS:mnt1_used:GAUGE:600:0:U",
			"DS:mnt1_free:GAUGE:600:0:U",
			"DS:mnt2_tot:GAUGE:600:0:U",
			"DS:mnt2_used:GAUGE:600:0:U",
			"DS:mnt2_free:GAUGE:600:0:U",
			"DS:mnt3_tot:GAUGE:600:0:U",
			"DS:mnt3_used:GAUGE:600:0:U",
			"DS:mnt3_free:GAUGE:600:0:U",
			"DS:mnt4_tot:GAUGE:600:0:U",
			"DS:mnt4_used:GAUGE:600:0:U",
			"DS:mnt4_free:GAUGE:600:0:U",
			"DS:mnt5_tot:GAUGE:600:0:U",
			"DS:mnt5_used:GAUGE:600:0:U",
			"DS:mnt5_free:GAUGE:600:0:U",
			"DS:mnt6_tot:GAUGE:600:0:U",
			"DS:mnt6_used:GAUGE:600:0:U",
			"DS:mnt6_free:GAUGE:600:0:U",
			"DS:mnt7_tot:GAUGE:600:0:U",
			"DS:mnt7_used:GAUGE:600:0:U",
			"DS:mnt7_free:GAUGE:600:0:U",
			"DS:read_cnt:COUNTER:600:0:U",
			"DS:write_cnt:COUNTER:600:0:U",
			"DS:read_sec:COUNTER:600:0:U",
			"DS:write_sec:COUNTER:600:0:U",
			"RRA:AVERAGE:0.5:1:288",
			"RRA:AVERAGE:0.5:6:336",
			"RRA:AVERAGE:0.5:12:744",
			"RRA:AVERAGE:0.5:288:365",
			"RRA:MIN:0.5:1:288",
			"RRA:MIN:0.5:6:336",
			"RRA:MIN:0.5:12:744",
			"RRA:MIN:0.5:288:365",
			"RRA:MAX:0.5:1:288",
			"RRA:MAX:0.5:6:336",
			"RRA:MAX:0.5:12:744",
			"RRA:MAX:0.5:288:365",
			"RRA:LAST:0.5:1:288",
			"RRA:LAST:0.5:6:336",
			"RRA:LAST:0.5:12:744",
			"RRA:LAST:0.5:288:365");
		my $err = RRDs::error;
		die("ERROR: while creating $DISK_RRD: $err\n") if $err;
	}
}

sub rrd_net {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $NET_RRD)) {
		RRDs::create($NET_RRD,
			"--step=60",
			"DS:net0_bytes_in:COUNTER:120:0:U",
			"DS:net0_bytes_out:COUNTER:120:0:U",
			"DS:net0_packs_in:COUNTER:120:0:U",
			"DS:net0_packs_out:COUNTER:120:0:U",
			"DS:net0_error_in:COUNTER:120:0:U",
			"DS:net0_error_out:COUNTER:120:0:U",
			"DS:net1_bytes_in:COUNTER:120:0:U",
			"DS:net1_bytes_out:COUNTER:120:0:U",
			"DS:net1_packs_in:COUNTER:120:0:U",
			"DS:net1_packs_out:COUNTER:120:0:U",
			"DS:net1_error_in:COUNTER:120:0:U",
			"DS:net1_error_out:COUNTER:120:0:U",
			"DS:net2_bytes_in:COUNTER:120:0:U",
			"DS:net2_bytes_out:COUNTER:120:0:U",
			"DS:net2_packs_in:COUNTER:120:0:U",
			"DS:net2_packs_out:COUNTER:120:0:U",
			"DS:net2_error_in:COUNTER:120:0:U",
			"DS:net2_error_out:COUNTER:120:0:U",
			"DS:net3_bytes_in:COUNTER:120:0:U",
			"DS:net3_bytes_out:COUNTER:120:0:U",
			"DS:net3_packs_in:COUNTER:120:0:U",
			"DS:net3_packs_out:COUNTER:120:0:U",
			"DS:net3_error_in:COUNTER:120:0:U",
			"DS:net3_error_out:COUNTER:120:0:U",
			"DS:net4_bytes_in:COUNTER:120:0:U",
			"DS:net4_bytes_out:COUNTER:120:0:U",
			"DS:net4_packs_in:COUNTER:120:0:U",
			"DS:net4_packs_out:COUNTER:120:0:U",
			"DS:net4_error_in:COUNTER:120:0:U",
			"DS:net4_error_out:COUNTER:120:0:U",
			"DS:net5_bytes_in:COUNTER:120:0:U",
			"DS:net5_bytes_out:COUNTER:120:0:U",
			"DS:net5_packs_in:COUNTER:120:0:U",
			"DS:net5_packs_out:COUNTER:120:0:U",
			"DS:net5_error_in:COUNTER:120:0:U",
			"DS:net5_error_out:COUNTER:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $NET_RRD: $err\n") if $err;
	}
}

sub rrd_servu {
# RRA: average data of 300*288 secs = 1 day
# RRA: average data of 300*6*336 secs = 1 week
# RRA: average data of 300*12*744 secs = 1 month
# RRA: average data of 300*288*365 secs = 1 year
# RRA: min data of 300*288 secs = 1 day
# RRA: min data of 300*6*336 secs = 1 week
# RRA: min data of 300*12*744 secs = 1 month
# RRA: min data of 300*288*365 secs = 1 year
# RRA: max data of 300*288 secs = 1 day
# RRA: max data of 300*6*336 secs = 1 week
# RRA: max data of 300*12*744 secs = 1 month
# RRA: max data of 300*288*365 secs = 1 year
# RRA: last data of 300*288 secs = 1 day
# RRA: last data of 300*6*336 secs = 1 week
# RRA: last data of 300*12*744 secs = 1 month
# RRA: last data of 300*288*365 secs = 1 year
	if(!(-e $SERVU_RRD)) {
		RRDs::create($SERVU_RRD,
			"--step=300",
			"DS:pop3:GAUGE:600:0:U",
			"DS:smtp:GAUGE:600:0:U",
			"DS:ssh:GAUGE:600:0:U",
			"DS:ftp:GAUGE:600:0:U",
			"DS:telnet:GAUGE:600:0:U",
			"DS:www:GAUGE:600:0:U",
			"DS:smb:GAUGE:600:0:U",
			"DS:virusmail:GAUGE:600:0:U",
			"DS:fax:GAUGE:600:0:U",
			"RRA:AVERAGE:0.5:1:288",
			"RRA:AVERAGE:0.5:6:336",
			"RRA:AVERAGE:0.5:12:744",
			"RRA:AVERAGE:0.5:288:365",
			"RRA:MIN:0.5:1:288",
			"RRA:MIN:0.5:6:336",
			"RRA:MIN:0.5:12:744",
			"RRA:MIN:0.5:288:365",
			"RRA:MAX:0.5:1:288",
			"RRA:MAX:0.5:6:336",
			"RRA:MAX:0.5:12:744",
			"RRA:MAX:0.5:288:365",
			"RRA:LAST:0.5:1:288",
			"RRA:LAST:0.5:6:336",
			"RRA:LAST:0.5:12:744",
			"RRA:LAST:0.5:288:365");
		my $err = RRDs::error;
		die("ERROR: while creating $SERVU_RRD: $err\n") if $err;
	}
}

sub rrd_mail {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $MAIL_RRD)) {
		RRDs::create($MAIL_RRD,
			"--step=60",
			"DS:mta_val01:COUNTER:120:0:U",
			"DS:mta_val02:COUNTER:120:0:U",
			"DS:mta_val03:COUNTER:120:0:U",
			"DS:mta_val04:COUNTER:120:0:U",
			"DS:mta_val05:COUNTER:120:0:U",
			"DS:mta_val06:COUNTER:120:0:U",
			"DS:mta_val07:COUNTER:120:0:U",
			"DS:mta_val08:COUNTER:120:0:U",
			"DS:mta_val09:COUNTER:120:0:U",
			"DS:mta_val10:COUNTER:120:0:U",
			"DS:mta_val11:GAUGE:120:0:U",
			"DS:mta_val12:GAUGE:120:0:U",
			"DS:mta_val13:GAUGE:120:0:U",
			"DS:mta_val14:GAUGE:120:0:U",
			"DS:mta_val15:GAUGE:120:0:U",
			"DS:val01:COUNTER:120:0:U",
			"DS:val02:COUNTER:120:0:U",
			"DS:val03:COUNTER:120:0:U",
			"DS:val04:COUNTER:120:0:U",
			"DS:val05:COUNTER:120:0:U",
			"DS:val06:GAUGE:120:0:U",
			"DS:val07:GAUGE:120:0:U",
			"DS:val08:GAUGE:120:0:U",
			"DS:val09:GAUGE:120:0:U",
			"DS:val10:GAUGE:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $MAIL_RRD: $err\n") if $err;
	}
}

sub rrd_port {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $PORT_RRD)) {
		RRDs::create($PORT_RRD,
			"--step=60",
			"DS:port01_bytes_in:COUNTER:120:0:U",
			"DS:port01_bytes_out:COUNTER:120:0:U",
			"DS:port02_bytes_in:COUNTER:120:0:U",
			"DS:port02_bytes_out:COUNTER:120:0:U",
			"DS:port03_bytes_in:COUNTER:120:0:U",
			"DS:port03_bytes_out:COUNTER:120:0:U",
			"DS:port04_bytes_in:COUNTER:120:0:U",
			"DS:port04_bytes_out:COUNTER:120:0:U",
			"DS:port05_bytes_in:COUNTER:120:0:U",
			"DS:port05_bytes_out:COUNTER:120:0:U",
			"DS:port06_bytes_in:COUNTER:120:0:U",
			"DS:port06_bytes_out:COUNTER:120:0:U",
			"DS:port07_bytes_in:COUNTER:120:0:U",
			"DS:port07_bytes_out:COUNTER:120:0:U",
			"DS:port08_bytes_in:COUNTER:120:0:U",
			"DS:port08_bytes_out:COUNTER:120:0:U",
			"DS:port09_bytes_in:COUNTER:120:0:U",
			"DS:port09_bytes_out:COUNTER:120:0:U",
			"DS:port10_bytes_in:COUNTER:120:0:U",
			"DS:port10_bytes_out:COUNTER:120:0:U",
			"DS:port11_bytes_in:COUNTER:120:0:U",
			"DS:port11_bytes_out:COUNTER:120:0:U",
			"DS:port12_bytes_in:COUNTER:120:0:U",
			"DS:port12_bytes_out:COUNTER:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $PORT_RRD: $err\n") if $err;
	}
}

sub rrd_user {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $USER_RRD)) {
		RRDs::create($USER_RRD,
			"--step=60",
			"DS:user_usr:GAUGE:120:0:U",
			"DS:user_smb:GAUGE:120:0:U",
			"DS:user_mac:GAUGE:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $USER_RRD: $err\n") if $err;
	}
}

sub rrd_nginx {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $NGINX_RRD)) {
		RRDs::create($NGINX_RRD,
			"--step=60",
			"DS:requests:ABSOLUTE:120:0:U",
			"DS:total:GAUGE:120:0:U",
			"DS:reading:GAUGE:120:0:U",
			"DS:writing:GAUGE:120:0:U",
			"DS:waiting:GAUGE:120:0:U",
			"DS:nginx_bytes_in:COUNTER:120:0:U",
			"DS:nginx_bytes_out:COUNTER:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $NGINX_RRD: $err\n") if $err;
	}
}

sub rrd_ints {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $INT_RRD)) {
		RRDs::create($INT_RRD,
			"--step=60",
			"DS:int_0:COUNTER:120:0:U",
			"DS:int_1:COUNTER:120:0:U",
			"DS:int_2:COUNTER:120:0:U",
			"DS:int_3:COUNTER:120:0:U",
			"DS:int_4:COUNTER:120:0:U",
			"DS:int_5:COUNTER:120:0:U",
			"DS:int_6:COUNTER:120:0:U",
			"DS:int_7:COUNTER:120:0:U",
			"DS:int_8:COUNTER:120:0:U",
			"DS:int_9:COUNTER:120:0:U",
			"DS:int_10:COUNTER:120:0:U",
			"DS:int_11:COUNTER:120:0:U",
			"DS:int_12:COUNTER:120:0:U",
			"DS:int_13:COUNTER:120:0:U",
			"DS:int_14:COUNTER:120:0:U",
			"DS:int_15:COUNTER:120:0:U",
			"DS:int_16:COUNTER:120:0:U",
			"DS:int_17:COUNTER:120:0:U",
			"DS:int_18:COUNTER:120:0:U",
			"DS:int_19:COUNTER:120:0:U",
			"DS:int_20:COUNTER:120:0:U",
			"DS:int_21:COUNTER:120:0:U",
			"DS:int_22:COUNTER:120:0:U",
			"DS:int_23:COUNTER:120:0:U",
			"DS:int_24:COUNTER:120:0:U",
			"DS:int_25:COUNTER:120:0:U",
			"DS:int_26:COUNTER:120:0:U",
			"DS:int_27:COUNTER:120:0:U",
			"DS:int_28:COUNTER:120:0:U",
			"DS:int_29:COUNTER:120:0:U",
			"DS:int_30:COUNTER:120:0:U",
			"DS:int_31:COUNTER:120:0:U",
			"DS:int_32:COUNTER:120:0:U",
			"DS:int_33:COUNTER:120:0:U",
			"DS:int_34:COUNTER:120:0:U",
			"DS:int_35:COUNTER:120:0:U",
			"DS:int_36:COUNTER:120:0:U",
			"DS:int_37:COUNTER:120:0:U",
			"DS:int_38:COUNTER:120:0:U",
			"DS:int_39:COUNTER:120:0:U",
			"DS:int_40:COUNTER:120:0:U",
			"DS:int_41:COUNTER:120:0:U",
			"DS:int_42:COUNTER:120:0:U",
			"DS:int_43:COUNTER:120:0:U",
			"DS:int_44:COUNTER:120:0:U",
			"DS:int_45:COUNTER:120:0:U",
			"DS:int_46:COUNTER:120:0:U",
			"DS:int_47:COUNTER:120:0:U",
			"DS:int_48:COUNTER:120:0:U",
			"DS:int_49:COUNTER:120:0:U",
			"DS:int_50:COUNTER:120:0:U",
			"DS:int_51:COUNTER:120:0:U",
			"DS:int_52:COUNTER:120:0:U",
			"DS:int_53:COUNTER:120:0:U",
			"DS:int_54:COUNTER:120:0:U",
			"DS:int_55:COUNTER:120:0:U",
			"DS:int_56:COUNTER:120:0:U",
			"DS:int_57:COUNTER:120:0:U",
			"DS:int_58:COUNTER:120:0:U",
			"DS:int_59:COUNTER:120:0:U",
			"DS:int_60:COUNTER:120:0:U",
			"DS:int_61:COUNTER:120:0:U",
			"DS:int_62:COUNTER:120:0:U",
			"DS:int_63:COUNTER:120:0:U",
			"DS:int_64:COUNTER:120:0:U",
			"DS:int_65:COUNTER:120:0:U",
			"DS:int_66:COUNTER:120:0:U",
			"DS:int_67:COUNTER:120:0:U",
			"DS:int_68:COUNTER:120:0:U",
			"DS:int_69:COUNTER:120:0:U",
			"DS:int_70:COUNTER:120:0:U",
			"DS:int_71:COUNTER:120:0:U",
			"DS:int_72:COUNTER:120:0:U",
			"DS:int_73:COUNTER:120:0:U",
			"DS:int_74:COUNTER:120:0:U",
			"DS:int_75:COUNTER:120:0:U",
			"DS:int_76:COUNTER:120:0:U",
			"DS:int_77:COUNTER:120:0:U",
			"DS:int_78:COUNTER:120:0:U",
			"DS:int_79:COUNTER:120:0:U",
			"DS:int_80:COUNTER:120:0:U",
			"DS:int_81:COUNTER:120:0:U",
			"DS:int_82:COUNTER:120:0:U",
			"DS:int_83:COUNTER:120:0:U",
			"DS:int_84:COUNTER:120:0:U",
			"DS:int_85:COUNTER:120:0:U",
			"DS:int_86:COUNTER:120:0:U",
			"DS:int_87:COUNTER:120:0:U",
			"DS:int_88:COUNTER:120:0:U",
			"DS:int_89:COUNTER:120:0:U",
			"DS:int_90:COUNTER:120:0:U",
			"DS:int_91:COUNTER:120:0:U",
			"DS:int_92:COUNTER:120:0:U",
			"DS:int_93:COUNTER:120:0:U",
			"DS:int_94:COUNTER:120:0:U",
			"DS:int_95:COUNTER:120:0:U",
			"DS:int_96:COUNTER:120:0:U",
			"DS:int_97:COUNTER:120:0:U",
			"DS:int_98:COUNTER:120:0:U",
			"DS:int_99:COUNTER:120:0:U",
			"DS:int_100:COUNTER:120:0:U",
			"DS:int_101:COUNTER:120:0:U",
			"DS:int_102:COUNTER:120:0:U",
			"DS:int_103:COUNTER:120:0:U",
			"DS:int_104:COUNTER:120:0:U",
			"DS:int_105:COUNTER:120:0:U",
			"DS:int_106:COUNTER:120:0:U",
			"DS:int_107:COUNTER:120:0:U",
			"DS:int_108:COUNTER:120:0:U",
			"DS:int_109:COUNTER:120:0:U",
			"DS:int_110:COUNTER:120:0:U",
			"DS:int_111:COUNTER:120:0:U",
			"DS:int_112:COUNTER:120:0:U",
			"DS:int_113:COUNTER:120:0:U",
			"DS:int_114:COUNTER:120:0:U",
			"DS:int_115:COUNTER:120:0:U",
			"DS:int_116:COUNTER:120:0:U",
			"DS:int_117:COUNTER:120:0:U",
			"DS:int_118:COUNTER:120:0:U",
			"DS:int_119:COUNTER:120:0:U",
			"DS:int_120:COUNTER:120:0:U",
			"DS:int_121:COUNTER:120:0:U",
			"DS:int_122:COUNTER:120:0:U",
			"DS:int_123:COUNTER:120:0:U",
			"DS:int_124:COUNTER:120:0:U",
			"DS:int_125:COUNTER:120:0:U",
			"DS:int_126:COUNTER:120:0:U",
			"DS:int_127:COUNTER:120:0:U",
			"DS:int_128:COUNTER:120:0:U",
			"DS:int_129:COUNTER:120:0:U",
			"DS:int_130:COUNTER:120:0:U",
			"DS:int_131:COUNTER:120:0:U",
			"DS:int_132:COUNTER:120:0:U",
			"DS:int_133:COUNTER:120:0:U",
			"DS:int_134:COUNTER:120:0:U",
			"DS:int_135:COUNTER:120:0:U",
			"DS:int_136:COUNTER:120:0:U",
			"DS:int_137:COUNTER:120:0:U",
			"DS:int_138:COUNTER:120:0:U",
			"DS:int_139:COUNTER:120:0:U",
			"DS:int_140:COUNTER:120:0:U",
			"DS:int_141:COUNTER:120:0:U",
			"DS:int_142:COUNTER:120:0:U",
			"DS:int_143:COUNTER:120:0:U",
			"DS:int_144:COUNTER:120:0:U",
			"DS:int_145:COUNTER:120:0:U",
			"DS:int_146:COUNTER:120:0:U",
			"DS:int_147:COUNTER:120:0:U",
			"DS:int_148:COUNTER:120:0:U",
			"DS:int_149:COUNTER:120:0:U",
			"DS:int_150:COUNTER:120:0:U",
			"DS:int_151:COUNTER:120:0:U",
			"DS:int_152:COUNTER:120:0:U",
			"DS:int_153:COUNTER:120:0:U",
			"DS:int_154:COUNTER:120:0:U",
			"DS:int_155:COUNTER:120:0:U",
			"DS:int_156:COUNTER:120:0:U",
			"DS:int_157:COUNTER:120:0:U",
			"DS:int_158:COUNTER:120:0:U",
			"DS:int_159:COUNTER:120:0:U",
			"DS:int_160:COUNTER:120:0:U",
			"DS:int_161:COUNTER:120:0:U",
			"DS:int_162:COUNTER:120:0:U",
			"DS:int_163:COUNTER:120:0:U",
			"DS:int_164:COUNTER:120:0:U",
			"DS:int_165:COUNTER:120:0:U",
			"DS:int_166:COUNTER:120:0:U",
			"DS:int_167:COUNTER:120:0:U",
			"DS:int_168:COUNTER:120:0:U",
			"DS:int_169:COUNTER:120:0:U",
			"DS:int_170:COUNTER:120:0:U",
			"DS:int_171:COUNTER:120:0:U",
			"DS:int_172:COUNTER:120:0:U",
			"DS:int_173:COUNTER:120:0:U",
			"DS:int_174:COUNTER:120:0:U",
			"DS:int_175:COUNTER:120:0:U",
			"DS:int_176:COUNTER:120:0:U",
			"DS:int_177:COUNTER:120:0:U",
			"DS:int_178:COUNTER:120:0:U",
			"DS:int_179:COUNTER:120:0:U",
			"DS:int_180:COUNTER:120:0:U",
			"DS:int_181:COUNTER:120:0:U",
			"DS:int_182:COUNTER:120:0:U",
			"DS:int_183:COUNTER:120:0:U",
			"DS:int_184:COUNTER:120:0:U",
			"DS:int_185:COUNTER:120:0:U",
			"DS:int_186:COUNTER:120:0:U",
			"DS:int_187:COUNTER:120:0:U",
			"DS:int_188:COUNTER:120:0:U",
			"DS:int_189:COUNTER:120:0:U",
			"DS:int_190:COUNTER:120:0:U",
			"DS:int_191:COUNTER:120:0:U",
			"DS:int_192:COUNTER:120:0:U",
			"DS:int_193:COUNTER:120:0:U",
			"DS:int_194:COUNTER:120:0:U",
			"DS:int_195:COUNTER:120:0:U",
			"DS:int_196:COUNTER:120:0:U",
			"DS:int_197:COUNTER:120:0:U",
			"DS:int_198:COUNTER:120:0:U",
			"DS:int_199:COUNTER:120:0:U",
			"DS:int_200:COUNTER:120:0:U",
			"DS:int_201:COUNTER:120:0:U",
			"DS:int_202:COUNTER:120:0:U",
			"DS:int_203:COUNTER:120:0:U",
			"DS:int_204:COUNTER:120:0:U",
			"DS:int_205:COUNTER:120:0:U",
			"DS:int_206:COUNTER:120:0:U",
			"DS:int_207:COUNTER:120:0:U",
			"DS:int_208:COUNTER:120:0:U",
			"DS:int_209:COUNTER:120:0:U",
			"DS:int_210:COUNTER:120:0:U",
			"DS:int_211:COUNTER:120:0:U",
			"DS:int_212:COUNTER:120:0:U",
			"DS:int_213:COUNTER:120:0:U",
			"DS:int_214:COUNTER:120:0:U",
			"DS:int_215:COUNTER:120:0:U",
			"DS:int_216:COUNTER:120:0:U",
			"DS:int_217:COUNTER:120:0:U",
			"DS:int_218:COUNTER:120:0:U",
			"DS:int_219:COUNTER:120:0:U",
			"DS:int_220:COUNTER:120:0:U",
			"DS:int_221:COUNTER:120:0:U",
			"DS:int_222:COUNTER:120:0:U",
			"DS:int_223:COUNTER:120:0:U",
			"DS:int_224:COUNTER:120:0:U",
			"DS:int_225:COUNTER:120:0:U",
			"DS:int_226:COUNTER:120:0:U",
			"DS:int_227:COUNTER:120:0:U",
			"DS:int_228:COUNTER:120:0:U",
			"DS:int_229:COUNTER:120:0:U",
			"DS:int_230:COUNTER:120:0:U",
			"DS:int_231:COUNTER:120:0:U",
			"DS:int_232:COUNTER:120:0:U",
			"DS:int_233:COUNTER:120:0:U",
			"DS:int_234:COUNTER:120:0:U",
			"DS:int_235:COUNTER:120:0:U",
			"DS:int_236:COUNTER:120:0:U",
			"DS:int_237:COUNTER:120:0:U",
			"DS:int_238:COUNTER:120:0:U",
			"DS:int_239:COUNTER:120:0:U",
			"DS:int_240:COUNTER:120:0:U",
			"DS:int_241:COUNTER:120:0:U",
			"DS:int_242:COUNTER:120:0:U",
			"DS:int_243:COUNTER:120:0:U",
			"DS:int_244:COUNTER:120:0:U",
			"DS:int_245:COUNTER:120:0:U",
			"DS:int_246:COUNTER:120:0:U",
			"DS:int_247:COUNTER:120:0:U",
			"DS:int_248:COUNTER:120:0:U",
			"DS:int_249:COUNTER:120:0:U",
			"DS:int_250:COUNTER:120:0:U",
			"DS:int_251:COUNTER:120:0:U",
			"DS:int_252:COUNTER:120:0:U",
			"DS:int_253:COUNTER:120:0:U",
			"DS:int_254:COUNTER:120:0:U",
			"DS:int_255:COUNTER:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $INT_RRD: $err\n") if $err;
	}
}

sub rrd_pc {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $PC_RRD)) {
		RRDs::create($PC_RRD,
			"--step=60",
			"DS:pc1_bytes_in:COUNTER:120:0:U",
			"DS:pc1_bytes_out:COUNTER:120:0:U",
			"DS:pc2_bytes_in:COUNTER:120:0:U",
			"DS:pc2_bytes_out:COUNTER:120:0:U",
			"DS:pc3_bytes_in:COUNTER:120:0:U",
			"DS:pc3_bytes_out:COUNTER:120:0:U",
			"DS:pc4_bytes_in:COUNTER:120:0:U",
			"DS:pc4_bytes_out:COUNTER:120:0:U",
			"DS:pc5_bytes_in:COUNTER:120:0:U",
			"DS:pc5_bytes_out:COUNTER:120:0:U",
			"DS:pc6_bytes_in:COUNTER:120:0:U",
			"DS:pc6_bytes_out:COUNTER:120:0:U",
			"DS:pc7_bytes_in:COUNTER:120:0:U",
			"DS:pc7_bytes_out:COUNTER:120:0:U",
			"DS:pc8_bytes_in:COUNTER:120:0:U",
			"DS:pc8_bytes_out:COUNTER:120:0:U",
			"DS:pc9_bytes_in:COUNTER:120:0:U",
			"DS:pc9_bytes_out:COUNTER:120:0:U",
			"DS:pc10_bytes_in:COUNTER:120:0:U",
			"DS:pc10_bytes_out:COUNTER:120:0:U",
			"DS:pc11_bytes_in:COUNTER:120:0:U",
			"DS:pc11_bytes_out:COUNTER:120:0:U",
			"DS:pc12_bytes_in:COUNTER:120:0:U",
			"DS:pc12_bytes_out:COUNTER:120:0:U",
			"DS:pc13_bytes_in:COUNTER:120:0:U",
			"DS:pc13_bytes_out:COUNTER:120:0:U",
			"DS:pc14_bytes_in:COUNTER:120:0:U",
			"DS:pc14_bytes_out:COUNTER:120:0:U",
			"DS:pc15_bytes_in:COUNTER:120:0:U",
			"DS:pc15_bytes_out:COUNTER:120:0:U",
			"DS:pc16_bytes_in:COUNTER:120:0:U",
			"DS:pc16_bytes_out:COUNTER:120:0:U",
			"DS:pc17_bytes_in:COUNTER:120:0:U",
			"DS:pc17_bytes_out:COUNTER:120:0:U",
			"DS:pc18_bytes_in:COUNTER:120:0:U",
			"DS:pc18_bytes_out:COUNTER:120:0:U",
			"DS:pc19_bytes_in:COUNTER:120:0:U",
			"DS:pc19_bytes_out:COUNTER:120:0:U",
			"DS:pc20_bytes_in:COUNTER:120:0:U",
			"DS:pc20_bytes_out:COUNTER:120:0:U",
			"DS:pc21_bytes_in:COUNTER:120:0:U",
			"DS:pc21_bytes_out:COUNTER:120:0:U",
			"DS:pc22_bytes_in:COUNTER:120:0:U",
			"DS:pc22_bytes_out:COUNTER:120:0:U",
			"DS:pc23_bytes_in:COUNTER:120:0:U",
			"DS:pc23_bytes_out:COUNTER:120:0:U",
			"DS:pc24_bytes_in:COUNTER:120:0:U",
			"DS:pc24_bytes_out:COUNTER:120:0:U",
			"DS:pc25_bytes_in:COUNTER:120:0:U",
			"DS:pc25_bytes_out:COUNTER:120:0:U",
			"DS:pc26_bytes_in:COUNTER:120:0:U",
			"DS:pc26_bytes_out:COUNTER:120:0:U",
			"DS:pc27_bytes_in:COUNTER:120:0:U",
			"DS:pc27_bytes_out:COUNTER:120:0:U",
			"DS:pc28_bytes_in:COUNTER:120:0:U",
			"DS:pc28_bytes_out:COUNTER:120:0:U",
			"DS:pc29_bytes_in:COUNTER:120:0:U",
			"DS:pc29_bytes_out:COUNTER:120:0:U",
			"DS:pc30_bytes_in:COUNTER:120:0:U",
			"DS:pc30_bytes_out:COUNTER:120:0:U",
			"DS:pc31_bytes_in:COUNTER:120:0:U",
			"DS:pc31_bytes_out:COUNTER:120:0:U",
			"DS:pc32_bytes_in:COUNTER:120:0:U",
			"DS:pc32_bytes_out:COUNTER:120:0:U",
			"DS:pc33_bytes_in:COUNTER:120:0:U",
			"DS:pc33_bytes_out:COUNTER:120:0:U",
			"DS:pc34_bytes_in:COUNTER:120:0:U",
			"DS:pc34_bytes_out:COUNTER:120:0:U",
			"DS:pc35_bytes_in:COUNTER:120:0:U",
			"DS:pc35_bytes_out:COUNTER:120:0:U",
			"DS:pc36_bytes_in:COUNTER:120:0:U",
			"DS:pc36_bytes_out:COUNTER:120:0:U",
			"DS:pc37_bytes_in:COUNTER:120:0:U",
			"DS:pc37_bytes_out:COUNTER:120:0:U",
			"DS:pc38_bytes_in:COUNTER:120:0:U",
			"DS:pc38_bytes_out:COUNTER:120:0:U",
			"DS:pc39_bytes_in:COUNTER:120:0:U",
			"DS:pc39_bytes_out:COUNTER:120:0:U",
			"DS:pc40_bytes_in:COUNTER:120:0:U",
			"DS:pc40_bytes_out:COUNTER:120:0:U",
			"DS:pc41_bytes_in:COUNTER:120:0:U",
			"DS:pc41_bytes_out:COUNTER:120:0:U",
			"DS:pc42_bytes_in:COUNTER:120:0:U",
			"DS:pc42_bytes_out:COUNTER:120:0:U",
			"DS:pc43_bytes_in:COUNTER:120:0:U",
			"DS:pc43_bytes_out:COUNTER:120:0:U",
			"DS:pc44_bytes_in:COUNTER:120:0:U",
			"DS:pc44_bytes_out:COUNTER:120:0:U",
			"DS:pc45_bytes_in:COUNTER:120:0:U",
			"DS:pc45_bytes_out:COUNTER:120:0:U",
			"DS:pc46_bytes_in:COUNTER:120:0:U",
			"DS:pc46_bytes_out:COUNTER:120:0:U",
			"DS:pc47_bytes_in:COUNTER:120:0:U",
			"DS:pc47_bytes_out:COUNTER:120:0:U",
			"DS:pc48_bytes_in:COUNTER:120:0:U",
			"DS:pc48_bytes_out:COUNTER:120:0:U",
			"DS:pc49_bytes_in:COUNTER:120:0:U",
			"DS:pc49_bytes_out:COUNTER:120:0:U",
			"DS:pc50_bytes_in:COUNTER:120:0:U",
			"DS:pc50_bytes_out:COUNTER:120:0:U",
			"DS:pc51_bytes_in:COUNTER:120:0:U",
			"DS:pc51_bytes_out:COUNTER:120:0:U",
			"DS:pc52_bytes_in:COUNTER:120:0:U",
			"DS:pc52_bytes_out:COUNTER:120:0:U",
			"DS:pc53_bytes_in:COUNTER:120:0:U",
			"DS:pc53_bytes_out:COUNTER:120:0:U",
			"DS:pc54_bytes_in:COUNTER:120:0:U",
			"DS:pc54_bytes_out:COUNTER:120:0:U",
			"DS:pc55_bytes_in:COUNTER:120:0:U",
			"DS:pc55_bytes_out:COUNTER:120:0:U",
			"DS:pc56_bytes_in:COUNTER:120:0:U",
			"DS:pc56_bytes_out:COUNTER:120:0:U",
			"DS:pc57_bytes_in:COUNTER:120:0:U",
			"DS:pc57_bytes_out:COUNTER:120:0:U",
			"DS:pc58_bytes_in:COUNTER:120:0:U",
			"DS:pc58_bytes_out:COUNTER:120:0:U",
			"DS:pc59_bytes_in:COUNTER:120:0:U",
			"DS:pc59_bytes_out:COUNTER:120:0:U",
			"DS:pc60_bytes_in:COUNTER:120:0:U",
			"DS:pc60_bytes_out:COUNTER:120:0:U",
			"DS:pc61_bytes_in:COUNTER:120:0:U",
			"DS:pc61_bytes_out:COUNTER:120:0:U",
			"DS:pc62_bytes_in:COUNTER:120:0:U",
			"DS:pc62_bytes_out:COUNTER:120:0:U",
			"DS:pc63_bytes_in:COUNTER:120:0:U",
			"DS:pc63_bytes_out:COUNTER:120:0:U",
			"DS:pc64_bytes_in:COUNTER:120:0:U",
			"DS:pc64_bytes_out:COUNTER:120:0:U",
			"DS:pc65_bytes_in:COUNTER:120:0:U",
			"DS:pc65_bytes_out:COUNTER:120:0:U",
			"DS:pc66_bytes_in:COUNTER:120:0:U",
			"DS:pc66_bytes_out:COUNTER:120:0:U",
			"DS:pc67_bytes_in:COUNTER:120:0:U",
			"DS:pc67_bytes_out:COUNTER:120:0:U",
			"DS:pc68_bytes_in:COUNTER:120:0:U",
			"DS:pc68_bytes_out:COUNTER:120:0:U",
			"DS:pc69_bytes_in:COUNTER:120:0:U",
			"DS:pc69_bytes_out:COUNTER:120:0:U",
			"DS:pc70_bytes_in:COUNTER:120:0:U",
			"DS:pc70_bytes_out:COUNTER:120:0:U",
			"DS:pc71_bytes_in:COUNTER:120:0:U",
			"DS:pc71_bytes_out:COUNTER:120:0:U",
			"DS:pc72_bytes_in:COUNTER:120:0:U",
			"DS:pc72_bytes_out:COUNTER:120:0:U",
			"DS:pc73_bytes_in:COUNTER:120:0:U",
			"DS:pc73_bytes_out:COUNTER:120:0:U",
			"DS:pc74_bytes_in:COUNTER:120:0:U",
			"DS:pc74_bytes_out:COUNTER:120:0:U",
			"DS:pc75_bytes_in:COUNTER:120:0:U",
			"DS:pc75_bytes_out:COUNTER:120:0:U",
			"DS:pc76_bytes_in:COUNTER:120:0:U",
			"DS:pc76_bytes_out:COUNTER:120:0:U",
			"DS:pc77_bytes_in:COUNTER:120:0:U",
			"DS:pc77_bytes_out:COUNTER:120:0:U",
			"DS:pc78_bytes_in:COUNTER:120:0:U",
			"DS:pc78_bytes_out:COUNTER:120:0:U",
			"DS:pc79_bytes_in:COUNTER:120:0:U",
			"DS:pc79_bytes_out:COUNTER:120:0:U",
			"DS:pc80_bytes_in:COUNTER:120:0:U",
			"DS:pc80_bytes_out:COUNTER:120:0:U",
			"DS:pc81_bytes_in:COUNTER:120:0:U",
			"DS:pc81_bytes_out:COUNTER:120:0:U",
			"DS:pc82_bytes_in:COUNTER:120:0:U",
			"DS:pc82_bytes_out:COUNTER:120:0:U",
			"DS:pc83_bytes_in:COUNTER:120:0:U",
			"DS:pc83_bytes_out:COUNTER:120:0:U",
			"DS:pc84_bytes_in:COUNTER:120:0:U",
			"DS:pc84_bytes_out:COUNTER:120:0:U",
			"DS:pc85_bytes_in:COUNTER:120:0:U",
			"DS:pc85_bytes_out:COUNTER:120:0:U",
			"DS:pc86_bytes_in:COUNTER:120:0:U",
			"DS:pc86_bytes_out:COUNTER:120:0:U",
			"DS:pc87_bytes_in:COUNTER:120:0:U",
			"DS:pc87_bytes_out:COUNTER:120:0:U",
			"DS:pc88_bytes_in:COUNTER:120:0:U",
			"DS:pc88_bytes_out:COUNTER:120:0:U",
			"DS:pc89_bytes_in:COUNTER:120:0:U",
			"DS:pc89_bytes_out:COUNTER:120:0:U",
			"DS:pc90_bytes_in:COUNTER:120:0:U",
			"DS:pc90_bytes_out:COUNTER:120:0:U",
			"DS:pc91_bytes_in:COUNTER:120:0:U",
			"DS:pc91_bytes_out:COUNTER:120:0:U",
			"DS:pc92_bytes_in:COUNTER:120:0:U",
			"DS:pc92_bytes_out:COUNTER:120:0:U",
			"DS:pc93_bytes_in:COUNTER:120:0:U",
			"DS:pc93_bytes_out:COUNTER:120:0:U",
			"DS:pc94_bytes_in:COUNTER:120:0:U",
			"DS:pc94_bytes_out:COUNTER:120:0:U",
			"DS:pc95_bytes_in:COUNTER:120:0:U",
			"DS:pc95_bytes_out:COUNTER:120:0:U",
			"DS:pc96_bytes_in:COUNTER:120:0:U",
			"DS:pc96_bytes_out:COUNTER:120:0:U",
			"DS:pc97_bytes_in:COUNTER:120:0:U",
			"DS:pc97_bytes_out:COUNTER:120:0:U",
			"DS:pc98_bytes_in:COUNTER:120:0:U",
			"DS:pc98_bytes_out:COUNTER:120:0:U",
			"DS:pc99_bytes_in:COUNTER:120:0:U",
			"DS:pc99_bytes_out:COUNTER:120:0:U",
			"DS:pc100_bytes_in:COUNTER:120:0:U",
			"DS:pc100_bytes_out:COUNTER:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $PC_RRD: $err\n") if $err;
	}
}

sub rrd_temp_ML310 {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $HPTEMP_RRD)) {
		RRDs::create($HPTEMP_RRD,
			"--step=60",
			"DS:sb:GAUGE:120:0:U",
			"DS:cpu1:GAUGE:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $HPTEMP_RRD: $err\n") if $err;
	}
}

sub rrd_temp_ML350 {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $HPTEMP_RRD)) {
		RRDs::create($HPTEMP_RRD,
			"--step=60",
			"DS:sb:GAUGE:120:0:U",
			"DS:cpu1:GAUGE:120:0:U",
			"DS:cpu2:GAUGE:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $HPTEMP_RRD: $err\n") if $err;
	}
}

sub rrd_temp_ML570 {
# RRA: average data of 60*1440 secs = 1 day
# RRA: average data of 60*30*336 secs = 1 week
# RRA: average data of 60*60*744 secs = 1 month
# RRA: average data of 60*1440*365 secs = 1 year
# RRA: min data of 60*1440 secs = 1 day
# RRA: min data of 60*30*336 secs = 1 week
# RRA: min data of 60*60*744 secs = 1 month
# RRA: min data of 60*1440*365 secs = 1 year
# RRA: max data of 60*1440 secs = 1 day
# RRA: max data of 60*30*336 secs = 1 week
# RRA: max data of 60*60*744 secs = 1 month
# RRA: max data of 60*1440*365 secs = 1 year
# RRA: last data of 60*1440 secs = 1 day
# RRA: last data of 60*30*336 secs = 1 week
# RRA: last data of 60*60*744 secs = 1 month
# RRA: last data of 60*1440*365 secs = 1 year
	if(!(-e $HPTEMP_RRD)) {
		RRDs::create($HPTEMP_RRD,
			"--step=60",
			"DS:cpu1:GAUGE:120:0:U",
			"DS:cpu2:GAUGE:120:0:U",
			"DS:cpu3:GAUGE:120:0:U",
			"DS:cpu4:GAUGE:120:0:U",
			"DS:sb1:GAUGE:120:0:U",
			"DS:sb2:GAUGE:120:0:U",
			"DS:scsi1:GAUGE:120:0:U",
			"DS:scsi2:GAUGE:120:0:U",
			"RRA:AVERAGE:0.5:1:1440",
			"RRA:AVERAGE:0.5:30:336",
			"RRA:AVERAGE:0.5:60:744",
			"RRA:AVERAGE:0.5:1440:365",
			"RRA:MIN:0.5:1:1440",
			"RRA:MIN:0.5:30:336",
			"RRA:MIN:0.5:60:744",
			"RRA:MIN:0.5:1440:365",
			"RRA:MAX:0.5:1:1440",
			"RRA:MAX:0.5:30:336",
			"RRA:MAX:0.5:60:744",
			"RRA:MAX:0.5:1440:365",
			"RRA:LAST:0.5:1:1440",
			"RRA:LAST:0.5:30:336",
			"RRA:LAST:0.5:60:744",
			"RRA:LAST:0.5:1440:365");
		my $err = RRDs::error;
		die("ERROR: while creating $HPTEMP_RRD: $err\n") if $err;
	}
}

# ==================================================
# 			UPDATE OPTION
# ==================================================

sub update {
	sleep(5);

	my ($min) = (localtime(time))[1];
	cpu() unless $ENABLE_CPU eq "N";
	kern() unless $ENABLE_KERN eq "N";
	proc() unless $ENABLE_PROC eq "N";
	lmtemp() unless $ENABLE_LMTEMP eq "N";
	if(!($min % 5)) {
		disk() unless $ENABLE_DISK eq "N";
	}
	net() unless $ENABLE_NET eq "N";
	if(!($min % 5)) {
		servu() unless $ENABLE_SERVU eq "N";
	}
	mail() unless $ENABLE_MAIL eq "N";
	port() unless $ENABLE_PORT eq "N";
	user() unless $ENABLE_USER eq "N";
	nginx() unless $ENABLE_NGINX eq "N";
	ints() unless $ENABLE_INT eq "N";
	if(scalar(@PC_LIST) != 0) {
		pc();
	}
	if($MACHINE eq "ML310") {
		temp_ML310();
	}
	elsif($MACHINE eq "ML330") {
		temp_ML350();
	}
	elsif($MACHINE eq "ML350") {
		temp_ML350();
	}
	elsif($MACHINE eq "ML570") {
		temp_ML570();
	}
}

sub cpu {
	my $LOAD1;
	my $LOAD5;
	my $LOAD15;
	my $NPROC;
	my $NPSLP;
	my $NPRUN;
	my $MBUF;
	my $MCAC;
	my $MFRE;
	my $MVAL1;
	my $MVAL2;

	my $dir;
	my $status;
	my $state;
	my $LASTTIME = 0;

	# Read last 15min of CPU load average from file
	if (open(LAST, $TMP_LASTCPU15)) {
		while (<LAST>) {
			if (/^cpu15 /) {
				my @tmp = split(' ', $_);
				(undef, $LASTTIME) = @tmp;
			}
		}
		close(LAST);
	}

	if(($OSTYPE eq "Linux-RHFC") ||
	  ($OSTYPE eq "Linux-Arch") ||
	  ($OSTYPE eq "Linux-Debian") ||
	  ($OSTYPE eq "Linux-Gentoo") ||
	  ($OSTYPE eq "Linux-Slack") ||
	  ($OSTYPE eq "Linux-SuSE") ||
	  ($OSTYPE eq "Linux-Generic")) {
		open(PROC, "/proc/loadavg");
		my @tmp = split(' ', <PROC>);
		close(PROC);
		($LOAD1, $LOAD5, $LOAD15, my $tmp1) = @tmp;
		($NPRUN, $NPSLP) = split('\/', $tmp1);
		chomp($NPSLP, $NPRUN);
		$NPROC = $NPSLP + $NPRUN;

		open(MEMINFO, "/proc/meminfo");
		while (<MEMINFO>) {
			if(/^Buffers:/) {
				my @tmp = split(' ', $_);
				$MBUF = $tmp[1];
			}
			if(/^Cached:/) {
				my @tmp = split(' ', $_);
				$MCAC = $tmp[1];
			}
			if(/^MemFree:/) {
				my @tmp = split(' ', $_);
				$MFRE = $tmp[1];
			}
		}
		close(MEMINFO);
		$MVAL1 = $MVAL2 = "";
	} elsif($OSTYPE eq "FreeBSD") {
		my $PAGE_SIZE;
		my @tmp;

		open(PROC, "/sbin/sysctl -n vm.loadavg |");
		@tmp = split(' ', <PROC>);
		close(PROC);
		(undef, $LOAD1, $LOAD5, $LOAD15) = @tmp;
		open(PROC, "/sbin/sysctl vm.vmtotal | grep 'Processes' |");
		@tmp = split(' ', <PROC>);
		close(PROC);
		(undef, undef, $NPRUN, undef, undef, undef, undef, undef, undef, undef, $NPSLP) = @tmp;
		$NPSLP =~ s/\)//;
		chomp($NPSLP, $NPRUN);
		$NPROC = $NPSLP + $NPRUN;
		$MBUF = `/sbin/sysctl -n vfs.bufspace`;
		$MCAC = `/sbin/sysctl -n vm.stats.vm.v_cache_count`;
		$MFRE = `/sbin/sysctl vm.vmtotal | grep "Free Memory" | awk -F ":" '{ print \$2 }' | sed s'/K//'`;

		chomp($MBUF);
		$MBUF = $MBUF / 1024;
		$PAGE_SIZE = `/sbin/sysctl -n vm.stats.vm.v_page_size`;
		$MVAL1 = `/sbin/sysctl -n vm.stats.vm.v_active_count`;
		$MVAL2 = `/sbin/sysctl -n vm.stats.vm.v_inactive_count`;
		chomp($PAGE_SIZE, $MCAC, $MVAL1, $MVAL2);
		$MCAC = ($PAGE_SIZE * $MCAC) / 1024;
		$MVAL1 = ($PAGE_SIZE * $MVAL1) / 1024;
		$MVAL2 = ($PAGE_SIZE * $MVAL2) / 1024;
	}
	my $rrdata = "N";

	chomp($LOAD1,
		$LOAD5,
		$LOAD15,
		$NPROC,
		$NPSLP,
		$NPRUN,
		$MBUF,
		$MCAC,
		$MFRE,
		$MVAL1,
		$MVAL2);

	# CPU alert
	if($ENABLE_ALERTS eq "Y") {
		if(!$ALERT_CPU_THRESHOLD || $LOAD15 < $ALERT_CPU_THRESHOLD) {
			unlink($TMP_LASTCPU15);
		} else {
			if(!$LASTTIME && open(LAST, "> ".$TMP_LASTCPU15)) {
				print(LAST "cpu15 " . time . "\n");
				close(LAST);
			}
			if($LASTTIME > 0 && (time - $LASTTIME) > $ALERT_CPU_TIMEINTVL) {
				if(-x $ALERT_CPU_SCRIPT) {
					system($ALERT_CPU_SCRIPT . " " .$ALERT_CPU_TIMEINTVL . " " . $ALERT_CPU_THRESHOLD . " " . $LOAD15);
				}
				if(open(LAST, "> ".$TMP_LASTCPU15)) {
					print(LAST "cpu15 " . time . "\n");
					close(LAST);
				}
			}
		}
	}

	$rrdata .= ":$LOAD1:$LOAD5:$LOAD15:$NPROC:$MBUF:$MCAC:$MFRE:$NPSLP:$NPRUN:$MVAL1:$MVAL2";
	RRDs::update($CPU_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $CPU_RRD: $err\n") if $err;
}

sub kern {
	my $USR;
	my $NIC;
	my $SYS;
	my $IDL;
	my $IOW;
	my $CS;
	
	my $LASTUSR = 0;
	my $LASTNIC = 0;
	my $LASTSYS = 0;
	my $LASTIDL = 0;
	my $LASTIOW = 0;

	# Read last kernel usage data from file
	if (open(LAST, $TMP_LASTKERN)) {
		while (<LAST>) {
			if (/^cpu /) {
				my @tmp = split(' ', $_);
				(undef, $LASTUSR, $LASTNIC, $LASTSYS, $LASTIDL, $LASTIOW) = @tmp;
			}
		}
		close(LAST);
	}

	if(($OSTYPE eq "Linux-RHFC") ||
	  ($OSTYPE eq "Linux-Arch") ||
	  ($OSTYPE eq "Linux-Debian") ||
	  ($OSTYPE eq "Linux-Gentoo") ||
	  ($OSTYPE eq "Linux-Slack") ||
	  ($OSTYPE eq "Linux-SuSE") ||
	  ($OSTYPE eq "Linux-Generic")) {
		open(STAT, "/proc/stat");
		while (<STAT>) {
			if(/^cpu /) {
				my @tmp = split(' ', $_);
				(undef, $USR, $NIC, $SYS, $IDL, $IOW) = @tmp;

				# Save current kernel usage data to file
				if (open(LAST, "> ".$TMP_LASTKERN)) {
					print(LAST $_);
					close(LAST);
				}
			}
			if(/^ctxt/) {
				my @tmp = split(' ', $_);
				(undef, $CS) = @tmp;
				last;
			}
		}
		close(STAT);
	} elsif($OSTYPE eq "FreeBSD") {
		my $cptime = `/sbin/sysctl -n kern.cp_time`;
		chomp($cptime);
		my @tmp = split(' ', $cptime);
		($USR, $NIC, $SYS, $IOW, $IDL) = @tmp;

		# Save current kernel usage data to file
		if (open(LAST, "> ".$TMP_LASTKERN)) {
			print(LAST "cpu  $USR $NIC $SYS $IDL $IOW\n");
			close(LAST);
		}

		$CS = `/sbin/sysctl -n vm.stats.sys.v_swtch`;
		chomp($CS);
	}
	$IOW = defined($IOW) ? $IOW : 0;	# for 2.4 Linux kernels
	$LASTIOW = defined($LASTIOW) ? $LASTIOW : 0; # for 2.4 Linux kernels
	my $rrdata = "N";

	if ($USR >= $LASTUSR && $NIC >= $LASTNIC && $SYS >= $LASTSYS && $IOW >= $LASTIOW && $IDL >= $LASTIDL) {
		my $dUSR = $USR - $LASTUSR;
		my $dNIC = $NIC - $LASTNIC;
		my $dSYS = $SYS - $LASTSYS;
		my $dIDL = $IDL - $LASTIDL;
		my $dIOW = $IOW - $LASTIOW;
		my $gtotal = $dUSR + $dNIC + $dSYS + $dIDL + $dIOW;
		$USR = ($dUSR * 100) / $gtotal;
		$NIC = ($dNIC * 100) / $gtotal;
		$SYS = ($dSYS * 100) / $gtotal;
		$IOW = ($dIOW * 100) / $gtotal;
	} else {
		$USR = "nan";
		$NIC = "nan";
		$SYS = "nan";
		$IOW = "nan";
	}

	$rrdata .= ":$USR:$NIC:$SYS:$IOW:$CS";
	RRDs::update($KERN_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $KERN_RRD: $err\n") if $err;
}

sub proc {
	my @cpu0;
	my @cpu1;
	my @cpu2;
	my @cpu3;
	my @cpu4;
	my @cpu5;
	my @cpu6;
	my @cpu7;
	my $gtotal;
	
	my @lastcpu0 = (0, 0, 0, 0, 0);
	my @lastcpu1 = (0, 0, 0, 0, 0);
	my @lastcpu2 = (0, 0, 0, 0, 0);
	my @lastcpu3 = (0, 0, 0, 0, 0);
	my @lastcpu4 = (0, 0, 0, 0, 0);
	my @lastcpu5 = (0, 0, 0, 0, 0);
	my @lastcpu6 = (0, 0, 0, 0, 0);
	my @lastcpu7 = (0, 0, 0, 0, 0);

	# Read last processor usage data from file
	if (open(LAST, $TMP_LASTPROC)) {
		while (<LAST>) {
			if (/^cpu0 /) {
				my @tmp = split(' ', $_);
				(undef, $lastcpu0[0], $lastcpu0[1], $lastcpu0[2], $lastcpu0[3], $lastcpu0[4]) = @tmp;
			}
			if (/^cpu1 /) {
				my @tmp = split(' ', $_);
				(undef, $lastcpu1[0], $lastcpu1[1], $lastcpu1[2], $lastcpu1[3], $lastcpu1[4]) = @tmp;
			}
			if (/^cpu2 /) {
				my @tmp = split(' ', $_);
				(undef, $lastcpu2[0], $lastcpu2[1], $lastcpu2[2], $lastcpu2[3], $lastcpu2[4]) = @tmp;
			}
			if (/^cpu3 /) {
				my @tmp = split(' ', $_);
				(undef, $lastcpu3[0], $lastcpu3[1], $lastcpu3[2], $lastcpu3[3], $lastcpu3[4]) = @tmp;
			}
			if (/^cpu4 /) {
				my @tmp = split(' ', $_);
				(undef, $lastcpu4[0], $lastcpu4[1], $lastcpu4[2], $lastcpu4[3], $lastcpu4[4]) = @tmp;
			}
			if (/^cpu5 /) {
				my @tmp = split(' ', $_);
				(undef, $lastcpu5[0], $lastcpu5[1], $lastcpu5[2], $lastcpu5[3], $lastcpu5[4]) = @tmp;
			}
			if (/^cpu6 /) {
				my @tmp = split(' ', $_);
				(undef, $lastcpu6[0], $lastcpu6[1], $lastcpu6[2], $lastcpu6[3], $lastcpu6[4]) = @tmp;
			}
			if (/^cpu7 /) {
				my @tmp = split(' ', $_);
				(undef, $lastcpu7[0], $lastcpu7[1], $lastcpu7[2], $lastcpu7[3], $lastcpu7[4]) = @tmp;
			}
		}
		close(LAST);
	}

	if(($OSTYPE eq "Linux-RHFC") ||
	  ($OSTYPE eq "Linux-Arch") ||
	  ($OSTYPE eq "Linux-Debian") ||
	  ($OSTYPE eq "Linux-Gentoo") ||
	  ($OSTYPE eq "Linux-Slack") ||
	  ($OSTYPE eq "Linux-SuSE") ||
	  ($OSTYPE eq "Linux-Generic")) {
		open(STAT, "/proc/stat");
		open(LAST, "> ".$TMP_LASTPROC);
		while (<STAT>) {
			if(/^cpu[0-7] /) {
				if(/^cpu0 /) {
					my @tmp = split(' ', $_);
					(undef, $cpu0[0], $cpu0[1], $cpu0[2], $cpu0[3], $cpu0[4]) = @tmp;
					print(LAST $_);
				}
				if(/^cpu1 /) {
					my @tmp = split(' ', $_);
					(undef, $cpu1[0], $cpu1[1], $cpu1[2], $cpu1[3], $cpu1[4]) = @tmp;
					print(LAST $_);
				}
				if(/^cpu2 /) {
					my @tmp = split(' ', $_);
					(undef, $cpu2[0], $cpu2[1], $cpu2[2], $cpu2[3], $cpu2[4]) = @tmp;
					print(LAST $_);
				}
				if(/^cpu3 /) {
					my @tmp = split(' ', $_);
					(undef, $cpu3[0], $cpu3[1], $cpu3[2], $cpu3[3], $cpu3[4]) = @tmp;
					print(LAST $_);
				}
				if(/^cpu4 /) {
					my @tmp = split(' ', $_);
					(undef, $cpu4[0], $cpu4[1], $cpu4[2], $cpu4[3], $cpu4[4]) = @tmp;
					print(LAST $_);
				}
				if(/^cpu5 /) {
					my @tmp = split(' ', $_);
					(undef, $cpu5[0], $cpu5[1], $cpu5[2], $cpu5[3], $cpu5[4]) = @tmp;
					print(LAST $_);
				}
				if(/^cpu6 /) {
					my @tmp = split(' ', $_);
					(undef, $cpu6[0], $cpu6[1], $cpu6[2], $cpu6[3], $cpu6[4]) = @tmp;
					print(LAST $_);
				}
				if(/^cpu7 /) {
					my @tmp = split(' ', $_);
					(undef, $cpu7[0], $cpu7[1], $cpu7[2], $cpu7[3], $cpu7[4]) = @tmp;
					print(LAST $_);
					last;
				}
			}
		}
		close(STAT);
		close(LAST);
	} elsif($OSTYPE eq "FreeBSD") {
		my $ncpu = `/sbin/sysctl -n hw.ncpu`;
		my $cptimes = `/sbin/sysctl -n kern.cp_times`;
		chomp($ncpu, $cptimes);
		my @tmp = split(' ', $cptimes);
		($cpu0[0], $cpu0[1], $cpu0[2], $cpu0[4], $cpu0[3],
		$cpu1[0], $cpu1[1], $cpu1[2], $cpu1[4], $cpu1[3],
		$cpu2[0], $cpu2[1], $cpu2[2], $cpu2[4], $cpu2[3],
		$cpu3[0], $cpu3[1], $cpu3[2], $cpu3[4], $cpu3[3],
		$cpu4[0], $cpu4[1], $cpu4[2], $cpu4[4], $cpu4[3],
		$cpu5[0], $cpu5[1], $cpu5[2], $cpu5[4], $cpu5[3],
		$cpu6[0], $cpu6[1], $cpu6[2], $cpu6[4], $cpu6[3],
		$cpu7[0], $cpu7[1], $cpu7[2], $cpu7[4], $cpu7[3]) = @tmp;
		undef(@cpu7) unless $ncpu > 7;
		undef(@cpu6) unless $ncpu > 6;
		undef(@cpu5) unless $ncpu > 5;
		undef(@cpu4) unless $ncpu > 4;
		undef(@cpu3) unless $ncpu > 3;
		undef(@cpu2) unless $ncpu > 2;
		undef(@cpu1) unless $ncpu > 1;
		open(LAST, "> ".$TMP_LASTPROC);
		if(1 <= $ncpu) {
			print(LAST "cpu0 $cpu0[0] $cpu0[1] $cpu0[2] $cpu0[3] $cpu0[4]\n");
		}
		if(2 <= $ncpu ) {
			print(LAST "cpu1 $cpu1[0] $cpu1[1] $cpu1[2] $cpu1[3] $cpu1[4]\n") unless !@cpu1;
		}
		if(3 <= $ncpu) {
			print(LAST "cpu2 $cpu2[0] $cpu2[1] $cpu2[2] $cpu2[3] $cpu2[4]\n") unless !@cpu2;
		}
		if(4 <= $ncpu) {
			print(LAST "cpu3 $cpu3[0] $cpu3[1] $cpu3[2] $cpu3[3] $cpu3[4]\n") unless !@cpu3;
		}
		if(5 <= $ncpu) {
			print(LAST "cpu4 $cpu4[0] $cpu4[1] $cpu4[2] $cpu4[3] $cpu4[4]\n") unless !@cpu4;
		}
		if(6 <= $ncpu) {
			print(LAST "cpu5 $cpu5[0] $cpu5[1] $cpu5[2] $cpu5[3] $cpu5[4]\n") unless !@cpu5;
		}
		if(7 <= $ncpu) {
			print(LAST "cpu6 $cpu6[0] $cpu6[1] $cpu6[2] $cpu6[3] $cpu6[4]\n") unless !@cpu6;
		}
		if(8 <= $ncpu) {
			print(LAST "cpu7 $cpu7[0] $cpu7[1] $cpu7[2] $cpu7[3] $cpu7[4]\n") unless !@cpu7;
		}
		close(LAST);
	}
	my $rrdata = "N";

	if(!@cpu0) {
		@cpu0 = (0,0,0,0,0);
	} else {
		$cpu0[4] = defined($cpu0[4]) ? $cpu0[4] : 0;	# 2.4 kernels
		$lastcpu0[4] = defined($lastcpu0[4]) ? $lastcpu0[4] : 0; # 2.4 kernels
		my @deltas = ($cpu0[0] - $lastcpu0[0], $cpu0[1] - $lastcpu0[1], $cpu0[2] - $lastcpu0[2], $cpu0[3] - $lastcpu0[3], $cpu0[4] - $lastcpu0[4]);
		$gtotal = $deltas[0] + $deltas[1] + $deltas[2] + $deltas[3] + $deltas[4];
		$cpu0[0] = ($deltas[0] * 100) / $gtotal;
		$cpu0[1] = ($deltas[1] * 100) / $gtotal;
		$cpu0[2] = ($deltas[2] * 100) / $gtotal;
		$cpu0[4] = ($deltas[4] * 100) / $gtotal;
	}
	if(!@cpu1) {
		@cpu1 = (0,0,0,0,0);
	} else {
		$cpu1[4] = defined($cpu1[4]) ? $cpu1[4] : 0;	# 2.4 kernels
		$lastcpu1[4] = defined($lastcpu1[4]) ? $lastcpu1[4] : 0; # 2.4 kernels
		my @deltas = ($cpu1[0] - $lastcpu1[0], $cpu1[1] - $lastcpu1[1], $cpu1[2] - $lastcpu1[2], $cpu1[3] - $lastcpu1[3], $cpu1[4] - $lastcpu1[4]);
		$gtotal = $deltas[0] + $deltas[1] + $deltas[2] + $deltas[3] + $deltas[4];
		$cpu1[0] = ($deltas[0] * 100) / $gtotal;
		$cpu1[1] = ($deltas[1] * 100) / $gtotal;
		$cpu1[2] = ($deltas[2] * 100) / $gtotal;
		$cpu1[4] = ($deltas[4] * 100) / $gtotal;
	}
	if(!@cpu2) {
		@cpu2 = (0,0,0,0,0);
	} else {
		$cpu2[4] = defined($cpu2[4]) ? $cpu2[4] : 0;	# 2.4 kernels
		$lastcpu2[4] = defined($lastcpu2[4]) ? $lastcpu2[4] : 0; # 2.4 kernels
		my @deltas = ($cpu2[0] - $lastcpu2[0], $cpu2[1] - $lastcpu2[1], $cpu2[2] - $lastcpu2[2], $cpu2[3] - $lastcpu2[3], $cpu2[4] - $lastcpu2[4]);
		$gtotal = $deltas[0] + $deltas[1] + $deltas[2] + $deltas[3] + $deltas[4];
		$cpu2[0] = ($deltas[0] * 100) / $gtotal;
		$cpu2[1] = ($deltas[1] * 100) / $gtotal;
		$cpu2[2] = ($deltas[2] * 100) / $gtotal;
		$cpu2[4] = ($deltas[4] * 100) / $gtotal;
	}
	if(!@cpu3) {
		@cpu3 = (0,0,0,0,0);
	} else {
		$cpu3[4] = defined($cpu3[4]) ? $cpu3[4] : 0;	# 2.4 kernels
		$lastcpu3[4] = defined($lastcpu3[4]) ? $lastcpu3[4] : 0; # 2.4 kernels
		my @deltas = ($cpu3[0] - $lastcpu3[0], $cpu3[1] - $lastcpu3[1], $cpu3[2] - $lastcpu3[2], $cpu3[3] - $lastcpu3[3], $cpu3[4] - $lastcpu3[4]);
		$gtotal = $deltas[0] + $deltas[1] + $deltas[2] + $deltas[3] + $deltas[4];
		$cpu3[0] = ($deltas[0] * 100) / $gtotal;
		$cpu3[1] = ($deltas[1] * 100) / $gtotal;
		$cpu3[2] = ($deltas[2] * 100) / $gtotal;
		$cpu3[4] = ($deltas[4] * 100) / $gtotal;
	}
	if(!@cpu4) {
		@cpu4 = (0,0,0,0,0);
	} else {
		$cpu4[4] = defined($cpu4[4]) ? $cpu4[4] : 0;	# 2.4 kernels
		$lastcpu4[4] = defined($lastcpu4[4]) ? $lastcpu4[4] : 0; # 2.4 kernels
		my @deltas = ($cpu4[0] - $lastcpu4[0], $cpu4[1] - $lastcpu4[1], $cpu4[2] - $lastcpu4[2], $cpu4[3] - $lastcpu4[3], $cpu4[4] - $lastcpu4[4]);
		$gtotal = $deltas[0] + $deltas[1] + $deltas[2] + $deltas[3] + $deltas[4];
		$cpu4[0] = ($deltas[0] * 100) / $gtotal;
		$cpu4[1] = ($deltas[1] * 100) / $gtotal;
		$cpu4[2] = ($deltas[2] * 100) / $gtotal;
		$cpu4[4] = ($deltas[4] * 100) / $gtotal;
	}
	if(!@cpu5) {
		@cpu5 = (0,0,0,0,0);
	} else {
		$cpu5[4] = defined($cpu5[4]) ? $cpu5[4] : 0;	# 2.4 kernels
		$lastcpu5[4] = defined($lastcpu5[4]) ? $lastcpu5[4] : 0; # 2.4 kernels
		my @deltas = ($cpu5[0] - $lastcpu5[0], $cpu5[1] - $lastcpu5[1], $cpu5[2] - $lastcpu5[2], $cpu5[3] - $lastcpu5[3], $cpu5[4] - $lastcpu5[4]);
		$gtotal = $deltas[0] + $deltas[1] + $deltas[2] + $deltas[3] + $deltas[4];
		$cpu5[0] = ($deltas[0] * 100) / $gtotal;
		$cpu5[1] = ($deltas[1] * 100) / $gtotal;
		$cpu5[2] = ($deltas[2] * 100) / $gtotal;
		$cpu5[4] = ($deltas[4] * 100) / $gtotal;
	}
	if(!@cpu6) {
		@cpu6 = (0,0,0,0,0);
	} else {
		$cpu6[4] = defined($cpu6[4]) ? $cpu6[4] : 0;	# 2.4 kernels
		$lastcpu6[4] = defined($lastcpu6[4]) ? $lastcpu6[4] : 0; # 2.4 kernels
		my @deltas = ($cpu6[0] - $lastcpu6[0], $cpu6[1] - $lastcpu6[1], $cpu6[2] - $lastcpu6[2], $cpu6[3] - $lastcpu6[3], $cpu6[4] - $lastcpu6[4]);
		$gtotal = $deltas[0] + $deltas[1] + $deltas[2] + $deltas[3] + $deltas[4];
		$cpu6[0] = ($deltas[0] * 100) / $gtotal;
		$cpu6[1] = ($deltas[1] * 100) / $gtotal;
		$cpu6[2] = ($deltas[2] * 100) / $gtotal;
		$cpu6[4] = ($deltas[4] * 100) / $gtotal;
	}
	if(!@cpu7) {
		@cpu7 = (0,0,0,0,0);
	} else {
		$cpu7[4] = defined($cpu7[4]) ? $cpu7[4] : 0;	# 2.4 kernels
		$lastcpu7[4] = defined($lastcpu7[4]) ? $lastcpu7[4] : 0; # 2.4 kernels
		my @deltas = ($cpu7[0] - $lastcpu7[0], $cpu7[1] - $lastcpu7[1], $cpu7[2] - $lastcpu7[2], $cpu7[3] - $lastcpu7[3], $cpu7[4] - $lastcpu7[4]);
		$gtotal = $deltas[0] + $deltas[1] + $deltas[2] + $deltas[3] + $deltas[4];
		$cpu7[0] = ($deltas[0] * 100) / $gtotal;
		$cpu7[1] = ($deltas[1] * 100) / $gtotal;
		$cpu7[2] = ($deltas[2] * 100) / $gtotal;
		$cpu7[4] = ($deltas[4] * 100) / $gtotal;
	}

	$rrdata .= ":$cpu0[0]:$cpu0[1]:$cpu0[2]:$cpu0[4]:$cpu1[0]:$cpu1[1]:$cpu1[2]:$cpu1[4]:$cpu2[0]:$cpu2[1]:$cpu2[2]:$cpu2[4]:$cpu3[0]:$cpu3[1]:$cpu3[2]:$cpu3[4]:$cpu4[0]:$cpu4[1]:$cpu4[2]:$cpu4[4]:$cpu5[0]:$cpu5[1]:$cpu5[2]:$cpu5[4]:$cpu6[0]:$cpu6[1]:$cpu6[2]:$cpu6[4]:$cpu7[0]:$cpu7[1]:$cpu7[2]:$cpu7[4]";
	RRDs::update($PROC_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $PROC_RRD: $err\n") if $err;
}


sub lmtemp {
	my $n;
	my $MB = 0;
	my @CPU;
	my @FAN;
	my @HD;
	my $cpu;
	my $fan;
	my $rrdata = "N";
	my $count;

	if(($OSTYPE eq "Linux-RHFC") ||
	  ($OSTYPE eq "Linux-Arch") ||
	  ($OSTYPE eq "Linux-Debian") ||
	  ($OSTYPE eq "Linux-Gentoo") ||
	  ($OSTYPE eq "Linux-Slack") ||
	  ($OSTYPE eq "Linux-SuSE") ||
	  ($OSTYPE eq "Linux-Generic")) {
		if(scalar(%SENSORS_LIST)) {
			no warnings;
	  		open(SENSORS, "sensors |");
			my @rawsensors = <SENSORS>;
			my $lines = @rawsensors;
			close(SENSORS);
			for($count = 0; $count < $lines; $count++) {
                               $_ = @rawsensors[$count];

				if($SENSORS_LIST{'MB'} && (/$SENSORS_LIST{'MB'}/) && (!/RPM/)) {
					my (undef, $tmp) = split(':', $_);
					if($tmp eq "\n") {
						$count++;
						$tmp = $rawsensors[$count];
					}
					($MB, undef) = split(' ', $tmp);
					$MB =~ s/[-]/./;
					$MB =~ s/[^0-9.]//g;
					$MB = int($MB);
				}
				for($n = 1; $n < 5; $n++) {
					$cpu = "CPU" . $n;
					if($SENSORS_LIST{$cpu} && (/$SENSORS_LIST{$cpu}/) && (!/RPM/)) {

						my (undef, $tmp) = split(':', $_);
						if($tmp eq "\n") {
							$count++;
							$tmp = $rawsensors[$count];
						}
						my ($value, undef) = split(' ', $tmp);
						$value =~ s/[-]/./;
						$value =~ s/[^0-9.]//g;
						if(int($value)>0) {
							$CPU[$n] = int($value);
						}
					}
				}
				for($n = 1; $n < 4; $n++) {
					$fan = "FAN" . $n;
					if($SENSORS_LIST{$fan} && (/$SENSORS_LIST{$fan}/) && (/RPM/)) {
						my (undef, $tmp) = split(':', $_);
						if($tmp eq "\n") {
							$count++;
							$tmp = $rawsensors[$count];
						}
						my ($value, undef) = split(' ', $tmp);
						$value =~ s/[^0-9]//g;
						if(int($value)>0) {
							$FAN[$n] = int($value);
						}
					}
				}
	
			}
		}
		$MB = defined($MB) ? $MB : 0;
		$CPU[1] = defined($CPU[1]) ? $CPU[1] : 0;
		$CPU[2] = defined($CPU[2]) ? $CPU[2] : 0;
		$CPU[3] = defined($CPU[3]) ? $CPU[3] : 0;
		$CPU[4] = defined($CPU[4]) ? $CPU[4] : 0;
		$FAN[1] = defined($FAN[1]) ? $FAN[1] : 0;
		$FAN[2] = defined($FAN[2]) ? $FAN[2] : 0;
		$FAN[3] = defined($FAN[3]) ? $FAN[3] : 0;
		$rrdata .= ":" . $MB . ":" . $CPU[1] . ":" . $CPU[2] . ":" . $CPU[3] . ":" . $CPU[4] . ":" . $FAN[1] . ":" . $FAN[2] . ":" . $FAN[3];
	  	for($n = 0; $n < 9; $n++) {
			$HD[$n] = 0;
			if(defined($HDDTEMP_LIST[$n])) {
				$HD[$n] = `hddtemp -wqn $HDDTEMP_LIST[$n]`;
				chomp($HD[$n]);
			}
			$rrdata .= ":$HD[$n]";
		}
	}
	RRDs::update($LMTEMP_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $LMTEMP_RRD: $err\n") if $err;
}

sub disk {
	my $n;
	my $kernel_branch = `uname -r | awk -F '.' '{ print \$1\".\"\$2 }'`;
	my @MNT_TOTAL;
	my @MNT_USED;
	my @MNT_FREE;
	my $rrdata = "N";
	my $root_disk;
	my $root_disk_p;
	my $READ_CNT = 0;
	my $READ_SEC = 0;
	my $WRITE_CNT = 0;
	my $WRITE_SEC = 0;

	my @tmp;
	my $mntpnt;
	my $mntpcnt;
	my $LASTTIME = 0;

	# Read last disk usage from file
	if (open(LAST, $TMP_LASTDISK)) {
		while (<LAST>) {
			if (/^dsk /) {
				my @tmp = split(' ', $_);
				(undef, $LASTTIME) = @tmp;
			}
		}
		close(LAST);
	}

	eval {
		alarm $TIMEOUT;
		@tmp = split(' ', `df -P / | tail -1`);
		alarm 0;
	};
	my (undef, $ROOT_TOTAL, $ROOT_USED, $ROOT_FREE, $ROOT_PCNT) = @tmp;

	my $SWAP_TOTAL;
	my $SWAP_USED;
	my $SWAP_FREE;
	if(($OSTYPE eq "Linux-RHFC") ||
	  ($OSTYPE eq "Linux-Arch") ||
	  ($OSTYPE eq "Linux-Debian") ||
	  ($OSTYPE eq "Linux-Gentoo") ||
	  ($OSTYPE eq "Linux-Slack") ||
	  ($OSTYPE eq "Linux-SuSE") ||
	  ($OSTYPE eq "Linux-Generic")) {
		@tmp = split(' ', `free | grep Swap`);
		(undef, $SWAP_TOTAL, $SWAP_USED, $SWAP_FREE) = @tmp;
	} elsif($OSTYPE eq "FreeBSD") {
		@tmp = split(' ', `/usr/sbin/swapinfo -k | tail -1`);
		(undef, $SWAP_TOTAL, $SWAP_USED, $SWAP_FREE) = @tmp;
	}

# We need to know exactly the name of the disk where the root filesystem
# resides. With a simple "df" command we only gets the partition name. That's
# why we need to make some steps before to obtain what we want.
	if(($OSTYPE eq "Linux-RHFC") ||
	  ($OSTYPE eq "Linux-Arch") ||
	  ($OSTYPE eq "Linux-Debian") ||
	  ($OSTYPE eq "Linux-Gentoo") ||
	  ($OSTYPE eq "Linux-Slack") ||
	  ($OSTYPE eq "Linux-SuSE") ||
	  ($OSTYPE eq "Linux-Generic") &&
	  $kernel_branch > 2.4) {
		my $lvm;
		my $lvm_disk;
		my $is_cciss;
		my $is_md;
		my $found = 0;
	
		eval {
			alarm $TIMEOUT;
			$root_disk = `df -P / | grep / | awk -F " " '{ print \$1 }'`;
			alarm 0;
		};
		chomp($root_disk);
		$root_disk_p = $root_disk;
		if($root_disk =~ m/cciss/) {
			$root_disk =~ s/[0-9]$//;
			$root_disk =~ s/[0-9]$//;
			$root_disk =~ s/.$//;
			$is_cciss = 1;
		}
		if($root_disk =~ m/\/dev\/evms\//) {
			$root_disk = `$EVMS disks $root_disk`;
			$found = 1;
		}
		if(!$found) {
		if(stat("/proc/mdstat")) {
			my $tmp;
			my $md_root_disk = $root_disk;
			$md_root_disk =~ s/\/dev\///;
			$md_root_disk =~ s/.*\///;
			$tmp = `grep -w $md_root_disk /proc/mdstat | awk -F " " '{ print \$1 }'`;
			chomp($tmp);
			if($tmp eq $md_root_disk) {
				$root_disk = $md_root_disk;
				$is_md = 1;
			}
		} else {
			$root_disk =~ s/.$//;
		}
		$root_disk =~ s/.*\///;
		$root_disk_p =~ s/.*\///;
	
		$lvm = $root_disk;
		$lvm =~ s/-.*//;
		if($lvm ne $root_disk) {	# probably LVM
			system("$PVS >/dev/null 2>&1");
			if($? == 0 && $lvm) {
				$lvm_disk = `$PVS --noheadings | grep $lvm | awk -F " " '{ print \$1 }' | tail -1`;
				chomp($lvm_disk);
				if(!($lvm_disk =~ m/md/)) {
					if(!($lvm_disk =~ m/cciss/)) {
						# LVM over any direct disk (/dev/hda1)
						$root_disk = `echo $lvm_disk | sed s'/.\$//'`;
						chomp($root_disk);
					} else {
						# LVM over a CCISS disk (/dev/cciss/c0d0)
						$root_disk = `echo $lvm_disk | sed s'/.\$//'`;
						chomp($root_disk);
						$root_disk = `echo $root_disk | sed s'/.\$//'`;
						chomp($root_disk);
					}
				} else {
					# LVM over RAID combination (/dev/md1)
					$root_disk = $lvm_disk;
					chomp($root_disk);
				}
				chomp($root_disk);
				$root_disk =~ s/.*\///;
				chomp($root_disk);
			}
		} else {
			if(!$is_cciss && !$is_md) {
				$root_disk =~ s/[0-9]$//;
				$root_disk =~ s/[0-9]$//;
			}

		}
		}
	} elsif($OSTYPE eq "FreeBSD") {
		eval {
			alarm $TIMEOUT;
			$root_disk = `df -P / | grep / | awk -F " " '{ print \$1 }'`;
			alarm 0;
		};
		chomp($root_disk);
		$root_disk =~ s/.*\///;		# removes /dev/
		$root_disk =~ s/...$//;		# removes part number
	}

	chomp($ROOT_TOTAL,
		$ROOT_USED,
		$ROOT_FREE,
		$SWAP_TOTAL,
		$SWAP_USED,
		$SWAP_FREE);
	$ROOT_PCNT =~ s/\%//;

	# DISK alert
	if($ENABLE_ALERTS eq "Y") {
		if(!$ALERT_DISK_THRESHOLD || $ROOT_PCNT < $ALERT_DISK_THRESHOLD) {
			unlink($TMP_LASTDISK);
		} else {
			if(!$LASTTIME && open(LAST, "> ".$TMP_LASTDISK)) {
				print(LAST "dsk " . time . "\n");
				close(LAST);
			}
			if($LASTTIME > 0 && (time - $LASTTIME) > $ALERT_DISK_TIMEINTVL) {
				if(-x $ALERT_DISK_SCRIPT) {
					system($ALERT_DISK_SCRIPT . " " . $ALERT_DISK_TIMEINTVL . " " . $ALERT_DISK_THRESHOLD . " " . $ROOT_PCNT);
				}
				if(open(LAST, "> ".$TMP_LASTDISK)) {
					print(LAST "dsk " . time . "\n");
					close(LAST);
				}
			}
		}
	}

	$rrdata .= ":$ROOT_TOTAL:$ROOT_USED:$ROOT_FREE:$SWAP_TOTAL:$SWAP_USED:$SWAP_FREE";
	for($n = 0; $n < 7; $n++) {
		$MNT_TOTAL[$n] = 0;
		$MNT_USED[$n] = 0;
		$MNT_FREE[$n] = 0;
		if($n < scalar(@MNT_LIST)) {
			eval {
				alarm $TIMEOUT;
				@tmp = split(' ', `df -P $MNT_LIST[$n] | tail -1`);
				alarm 0;
			};
			(undef, $MNT_TOTAL[$n], $MNT_USED[$n], $MNT_FREE[$n], undef, $mntpnt) = @tmp;
			if(!($mntpnt eq $MNT_LIST[$n])) {
				$MNT_TOTAL[$n] = $MNT_FREE[$n] = 100;
				$MNT_USED[$n] = 0;
			}
		}
		chomp($MNT_TOTAL[$n]);
		chomp($MNT_USED[$n]);
		chomp($MNT_FREE[$n]);
		$rrdata .= ":$MNT_TOTAL[$n]:$MNT_USED[$n]:$MNT_FREE[$n]";
	}

	if(($OSTYPE eq "Linux-RHFC") ||
	  ($OSTYPE eq "Linux-Arch") ||
	  ($OSTYPE eq "Linux-Debian") ||
	  ($OSTYPE eq "Linux-Gentoo") ||
	  ($OSTYPE eq "Linux-Slack") ||
	  ($OSTYPE eq "Linux-SuSE") ||
	  ($OSTYPE eq "Linux-Generic")) {
		if($kernel_branch > 2.4) {
			if(stat("/proc/diskstats")) {
				@tmp = split(' ', `cat /proc/diskstats | grep -w $root_disk`);
				if(!@tmp) {
					@tmp = split(' ', `cat /proc/diskstats | grep -w $root_disk_p`);
				}
				(undef, undef, undef, $READ_CNT, undef, $READ_SEC, undef, $WRITE_CNT, undef, $WRITE_SEC) = @tmp;
			}
		} else {
			$READ_CNT = `cat /proc/stat | grep "disk_io" | awk -F " " '{ print \$NF }' | awk -F ":" '{ print \$NF }' | awk -F "," '{ print \$2 }'`;
			$WRITE_CNT = `cat /proc/stat | grep "disk_io" | awk -F " " '{ print \$NF }' | awk -F ":" '{ print \$NF }' | awk -F "," '{ print \$4 }'`;
			$READ_SEC = `cat /proc/stat | grep "disk_io" | awk -F " " '{ print \$NF }' | awk -F ":" '{ print \$NF }' | awk -F "," '{ print \$3 }'`;
			$WRITE_SEC = `cat /proc/stat | grep "disk_io" | awk -F " " '{ print \$NF }' | awk -F ":" '{ print \$NF }' | awk -F "," '{ print \$5 }' | awk -F ")" '{ print \$1 }'`;
		}
	} elsif($OSTYPE eq "FreeBSD") {
		@tmp = split(' ', `iostat -xI $root_disk | grep -w $root_disk`);
		if(@tmp) {
			(undef, $READ_CNT, $WRITE_CNT, $READ_SEC, $WRITE_SEC) = @tmp;
			$READ_CNT = int($READ_CNT);
			$WRITE_CNT = int($WRITE_CNT);
			$READ_SEC = int($READ_SEC);
			$WRITE_SEC = int($WRITE_SEC);
		} else {
			@tmp = split(' ', `/usr/sbin/iostat -dI | tail -1`);
			(undef, $READ_CNT, $READ_SEC) = @tmp;
			$WRITE_CNT = "";
			$WRITE_SEC = "";
			chomp($READ_SEC);
			$READ_SEC = int($READ_SEC);
		}
	}

	chomp($READ_CNT, $READ_SEC, $WRITE_CNT, $WRITE_SEC);
	$rrdata .= ":$READ_CNT:$WRITE_CNT:$READ_SEC:$WRITE_SEC";
	RRDs::update($DISK_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $DISK_RRD: $err\n") if $err;
}

sub net {
	my $n;
	my @NET_BYTES_IN;
	my @NET_BYTES_OUT;
	my @NET_PACKS_IN;
	my @NET_PACKS_OUT;
	my @NET_ERROR_IN;
	my @NET_ERROR_OUT;
	my $rrdata = "N";

	for($n = 0; $n < 6 ; $n++) {
		$NET_BYTES_IN[$n] = 0;
		$NET_BYTES_OUT[$n] = 0;
		$NET_PACKS_IN[$n] = 0;
		$NET_PACKS_OUT[$n] = 0;
		$NET_ERROR_IN[$n] = 0;
		$NET_ERROR_OUT[$n] = 0;
		if($n < scalar(@NET_LIST)) {
			if(($OSTYPE eq "Linux-RHFC") ||
	  		  ($OSTYPE eq "Linux-Arch") ||
	  		  ($OSTYPE eq "Linux-Debian") ||
	  		  ($OSTYPE eq "Linux-Gentoo") ||
	  		  ($OSTYPE eq "Linux-Slack") ||
			  ($OSTYPE eq "Linux-SuSE") ||
	  		  ($OSTYPE eq "Linux-Generic")) {
				open(NET, "cat /proc/net/dev | grep $NET_LIST[$n] |");
				my (undef, $tmp) = split(':', <NET>);
				close(NET);
				if($tmp) {
					($NET_BYTES_IN[$n], $NET_PACKS_IN[$n], $NET_ERROR_IN[$n], undef, undef, undef, undef, undef, $NET_BYTES_OUT[$n], $NET_PACKS_OUT[$n], $NET_ERROR_OUT[$n]) = split(' ', $tmp);
				}
			} elsif($OSTYPE eq "FreeBSD") {
				open(NET, "netstat -nibd | grep Link | grep $NET_LIST[$n] | head -1 |");
				(undef, undef, undef, undef, $NET_PACKS_IN[$n], $NET_ERROR_IN[$n], $NET_BYTES_IN[$n], $NET_PACKS_OUT[$n], $NET_ERROR_OUT[$n], $NET_BYTES_OUT[$n]) = split(' ', <NET>);
				close(NET);
			}
		}
		chomp($NET_BYTES_IN[$n],
			$NET_BYTES_OUT[$n],
			$NET_PACKS_IN[$n],
			$NET_PACKS_OUT[$n],
			$NET_ERROR_IN[$n],
			$NET_ERROR_OUT[$n]);
		$rrdata .= ":$NET_BYTES_IN[$n]:$NET_BYTES_OUT[$n]:$NET_PACKS_IN[$n]:$NET_PACKS_OUT[$n]:$NET_ERROR_IN[$n]:$NET_ERROR_OUT[$n]";
	}

	RRDs::update($NET_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $NET_RRD: $err\n") if $err;
}

sub servu {
	my $FAX_LOG = "/var/spool/hylafax/etc/xferfaxlog";
	my $DATE_STR = `date '+\%b \%e'`;
	my $DATE_WWW = `date +\%d/\%b/\%Y`;
	my $DATE_FAX = `date +\%m/\%d/\%y`;
	chomp($DATE_STR, $DATE_WWW, $DATE_FAX);
	my $POP3 = 0;
	my $FTP = 0;
	my $TELNET = 0;
	my $SSH = 0;
	my $SMTP = 0;
	my $VIRUSMAIL = 0;
	my $WWW = 0;
	my $SMB = 0;
	my $rrdata = "N";

	if( -r $SECURE_LOG{$OSTYPE} ) {
		open(LISTING, "< $SECURE_LOG{$OSTYPE}");
		while(<LISTING>) {
			if(/$DATE_STR/) {
				if(/sshd/ && /Accepted /) {
					$SSH++;
				}
				if(($OSTYPE eq "Linux-RHFC") ||
	  			  ($OSTYPE eq "Linux-Arch") ||
	  			  ($OSTYPE eq "Linux-Debian") ||
	  			  ($OSTYPE eq "Linux-Gentoo") ||
	  			  ($OSTYPE eq "Linux-Slack") ||
				  ($OSTYPE eq "Linux-SuSE") ||
	  		  	  ($OSTYPE eq "Linux-Generic")) {
					if(/START: pop3/) {
						$POP3++;
					}
					if(/START: ftp/) {
						$FTP++;
					}
					if(/START: telnet/) {
						$TELNET++;
					}
				} elsif($OSTYPE eq "FreeBSD") {

					if(/login:/ && /login from /) {
						$TELNET++;
					}
				}
			}
		}
		close(LISTING);
	}

	if( -r $MAIL_LOG{$OSTYPE} ) {
		open(LISTING, "< $MAIL_LOG{$OSTYPE}");
		while(<LISTING>) {
			if(/$DATE_STR/) {
				if(/to=/ && /stat(us)?=sent/i) {
					$SMTP++;	
				}
				if(/MailScanner/ && /Virus Scanning:/ && /Found/ && /viruses/) {
					$VIRUSMAIL++;
				}
			}
		}
		close(LISTING);
	}

	if( -r $HTTP_LOG{$OSTYPE} ) {
		open(LISTING, "< $HTTP_LOG{$OSTYPE}");
		while(<LISTING>) {
			if(/$DATE_WWW/) {
				$WWW++;	
			}
		}
		close(LISTING);
	} else {
		print "can't read ", $HTTP_LOG{$OSTYPE}, "\n";
	}

	if(stat($SMBSTATUS{$OSTYPE})) {
		my $SMB_L = 0;
		my $SMB_S = 0;
		if($SAMBAVER eq "2") {
			$SMB_L = `$SMBSTATUS{$OSTYPE} -L 2>/dev/null | wc -l`;
			$SMB_L = $SMB_L - 5;
			$SMB_L = $SMB_L < 0 ? 0 : $SMB_L;
			$SMB_S = `$SMBSTATUS{$OSTYPE} -S 2>/dev/null | wc -l`;
			$SMB_S = $SMB_S - 5;
			$SMB_S = $SMB_S < 0 ? 0 : $SMB_S;
		}
		elsif($SAMBAVER eq "3") {
			$SMB_L = `$SMBSTATUS{$OSTYPE} -L 2>/dev/null | wc -l`;
			$SMB_L = $SMB_L - 4;
			$SMB_L = $SMB_L < 0 ? 0 : $SMB_L;
			$SMB_S = `$SMBSTATUS{$OSTYPE} -S 2>/dev/null | grep -iA 9999 "Service      pid     machine       Connected at" | wc -l`;
			$SMB_S = $SMB_S - 3;
			$SMB_S = $SMB_S < 0 ? 0 : $SMB_S;
		}
		$SMB=$SMB_L + $SMB_S;
	}

	my $FAX = 0;
	if( -r $FAX_LOG ) {
		open(LISTING, "< $FAX_LOG");
		while(<LISTING>) {
			if(/$DATE_FAX/) {
				$FAX++;
			}
		}
		close(LISTING);
	}
	chomp($POP3,
		$SMTP,
		$SSH,
		$FTP,
		$TELNET,
		$WWW,
		$SMB,
		$VIRUSMAIL,
		$FAX);
	$rrdata .= ":$POP3:$SMTP:$SSH:$FTP:$TELNET:$WWW:$SMB:$VIRUSMAIL:$FAX";
	RRDs::update($SERVU_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $SERVU_RRD: $err\n") if $err;
}

sub mail {
	my $seek_pos;
	my $logsize;
	my $n;
	my $in_conn;
	my $out_conn;
	my $recvd;
	my $delvd;
	my $bytes_recvd;
	my $bytes_delvd;
	my $rejected;
	my $queued;
	my $spam;
	my $virus;
	my $gl_records;
	my $gl_greylisted;
	my $gl_whitelisted;
	my @mta;
	my @gen;
	my $rrdata = "N";

	$in_conn = $out_conn = $recvd = $delvd = $bytes_recvd = $bytes_delvd = 0;
	$rejected = $queued = 0;
	$spam = $virus = 0;
	$gl_records = $gl_greylisted = $gl_whitelisted = 0;

	if($MTA eq "sendmail") {
		if(open(STATS, "mailstats -P |")) {
			while(<STATS>) {
				if(/^ T/) {
					my @line = split(' ', $_);
					(undef, $recvd, $bytes_recvd, $delvd, $bytes_delvd) = @line;
				}
				if(/^ C/) {
					my @line = split(' ', $_);
					(undef, $in_conn, $out_conn, $rejected) = @line;
				}
			}
			close(STATS);
			$bytes_recvd *= 1024;
			$bytes_delvd *= 1024;
		}
		if(open(STATS, "mailq |")) {
			while(<STATS>) {
				if(/Total requests:/) {
					my @line = split(' ', $_);
					(undef, undef, $queued) = @line;
				}
			}
			close(STATS);
		}
	}

	if($GREYLIST eq "milter-greylist") {
		if(-r $MILTER_GL{$OSTYPE}) {
			open(LOG, $MILTER_GL{$OSTYPE});
			seek(LOG, -80, 2) or die "Couldn't seek to the end ($MILTER_GL{$OSTYPE}): $!\n";
			while(<LOG>) {
				if(/^# Summary:/) {
					(undef, undef, $gl_records, undef, $gl_greylisted, undef, $gl_whitelisted) = split(' ', $_);
				}
			}
			close(LOG);
		}
	}

	for($n = 0; $n < 15; $n++) {
		$mta[$n] = 0;
		if($n < 10) {
			$gen[$n] = 0;
		}
	}
	if(open(LAST, $TMP_MAILLOG)) {
		chomp(my $line = <LAST>);
		($seek_pos, @mta[0..15-1], @gen[0..10-1]) = split(' ', $line);
		close(LAST);
	}
	$seek_pos = defined($seek_pos) ? int($seek_pos) : 0;

	if(-r $MAIL_LOG{$OSTYPE}) {
		$spam = int($mta[8]);
		$virus = int($mta[9]);
		open(LOG, $MAIL_LOG{$OSTYPE});
		seek(LOG, 0, 2) or die "Couldn't seek to the end ($MAIL_LOG{$OSTYPE}): $!\n";
		$logsize = tell(LOG);
		if($logsize < $seek_pos) {
			$seek_pos = 0;
		}
		seek(LOG, $seek_pos, 0) or die "Couldn't seek to $seek_pos ($MAIL_LOG{$OSTYPE}): $!\n";
		while(<LOG>) {
			my @line;
			if(/MailScanner/ && /Spam Checks:/ && /Found/
			&& /spam messages/) {
				@line = split(' ', $_);
				$spam += int($line[8]);
			}
			if(/MailScanner/ && /Virus Scanning:/ && /Found/
			&& /viruses/) {
				@line = split(' ', $_);
				$virus += int($line[8]);
			}
		}
		close(LOG);
	}

	$mta[0] = int($in_conn);
	$mta[1] = int($out_conn);
	$mta[2] = int($recvd);
	$mta[3] = int($delvd);
	$mta[4] = int($bytes_recvd);
	$mta[5] = int($bytes_delvd);
	$mta[6] = int($rejected);
	$mta[8] = int($spam);
	$mta[9] = int($virus);
	$mta[11] = int($queued);

	$gen[6] = int($gl_records);
	$gen[7] = int($gl_greylisted);
	$gen[8] = int($gl_whitelisted);

	if(open(LAST, "> " . $TMP_MAILLOG)) {
		print(LAST join(" ", $logsize, @mta, @gen) . "\n");
		close(LAST);
	}

	for($n = 0; $n < 15; $n++) {
		$rrdata .= ":" . int($mta[$n]);
	}
	for($n = 0; $n < 10; $n++) {
		$rrdata .= ":" . int($gen[$n]);
	}

	RRDs::update($MAIL_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $MAIL_RRD: $err\n") if $err;
}

sub port {
	my $PORT01_IN = "";
	my $PORT01_OUT = "";
	my $PORT02_IN = "";
	my $PORT02_OUT = "";
	my $PORT03_IN = "";
	my $PORT03_OUT = "";
	my $PORT04_IN = "";
	my $PORT04_OUT = "";
	my $PORT05_IN = "";
	my $PORT05_OUT = "";
	my $PORT06_IN = "";
	my $PORT06_OUT = "";
	my $PORT07_IN = "";
	my $PORT07_OUT = "";
	my $PORT08_IN = "";
	my $PORT08_OUT = "";
	my $PORT09_IN = "";
	my $PORT09_OUT = "";
	my $PORT10_IN = "";
	my $PORT10_OUT = "";
	my $PORT11_IN = "";
	my $PORT11_OUT = "";
	my $PORT12_IN = "";
	my $PORT12_OUT = "";
	my $rrdata = "N";

	if($IPTABLES) {
		open(IPTABLES, "$IPTABLES -nxvL INPUT |");
		while(<IPTABLES>) {
			if(/PORT01_IN/) {
				(undef, $PORT01_IN) = split(' ', $_);
			}
			if(/PORT02_IN/) {
				(undef, $PORT02_IN) = split(' ', $_);
			}
			if(/PORT03_IN/) {
				(undef, $PORT03_IN) = split(' ', $_);
			}
			if(/PORT04_IN/) {
				(undef, $PORT04_IN) = split(' ', $_);
			}
			if(/PORT05_IN/) {
				(undef, $PORT05_IN) = split(' ', $_);
			}
			if(/PORT06_IN/) {
				(undef, $PORT06_IN) = split(' ', $_);
			}
			if(/PORT07_IN/) {
				(undef, $PORT07_IN) = split(' ', $_);
			}
			if(/PORT08_IN/) {
				(undef, $PORT08_IN) = split(' ', $_);
			}
			if(/PORT09_IN/) {
				(undef, $PORT09_IN) = split(' ', $_);
			}
			if(/PORT10_IN/) {
				(undef, $PORT10_IN) = split(' ', $_);
			}
			if(/PORT11_IN/) {
				(undef, $PORT11_IN) = split(' ', $_);
			}
			if(/PORT12_IN/) {
				(undef, $PORT12_IN) = split(' ', $_);
			}
		}
		close(IPTABLES);
		open(IPTABLES, "$IPTABLES -nxvL OUTPUT |");
		while(<IPTABLES>) {
			if(/PORT01_OUT/) {
				(undef, $PORT01_OUT) = split(' ', $_);
			}
			if(/PORT02_OUT/) {
				(undef, $PORT02_OUT) = split(' ', $_);
			}
			if(/PORT03_OUT/) {
				(undef, $PORT03_OUT) = split(' ', $_);
			}
			if(/PORT04_OUT/) {
				(undef, $PORT04_OUT) = split(' ', $_);
			}
			if(/PORT05_OUT/) {
				(undef, $PORT05_OUT) = split(' ', $_);
			}
			if(/PORT06_OUT/) {
				(undef, $PORT06_OUT) = split(' ', $_);
			}
			if(/PORT07_OUT/) {
				(undef, $PORT07_OUT) = split(' ', $_);
			}
			if(/PORT08_OUT/) {
				(undef, $PORT08_OUT) = split(' ', $_);
			}
			if(/PORT09_OUT/) {
				(undef, $PORT09_OUT) = split(' ', $_);
			}
			if(/PORT10_OUT/) {
				(undef, $PORT10_OUT) = split(' ', $_);
			}
			if(/PORT11_OUT/) {
				(undef, $PORT11_OUT) = split(' ', $_);
			}
			if(/PORT12_OUT/) {
				(undef, $PORT12_OUT) = split(' ', $_);
			}
		}
		close(IPTABLES);
	}

	chomp($PORT01_IN,
		$PORT01_OUT,
		$PORT02_IN,
		$PORT02_OUT,
		$PORT03_IN,
		$PORT03_OUT,
		$PORT04_IN,
		$PORT04_OUT,
		$PORT05_IN,
		$PORT05_OUT,
		$PORT06_IN,
		$PORT06_OUT,
		$PORT07_IN,
		$PORT07_OUT,
		$PORT08_IN,
		$PORT08_OUT,
		$PORT09_IN,
		$PORT09_OUT,
		$PORT10_IN,
		$PORT10_OUT,
		$PORT11_IN,
		$PORT11_OUT,
		$PORT12_IN,
		$PORT12_OUT);
	$rrdata .= ":$PORT01_IN:$PORT01_OUT:$PORT02_IN:$PORT02_OUT:$PORT03_IN:$PORT03_OUT:$PORT04_IN:$PORT04_OUT:$PORT05_IN:$PORT05_OUT:$PORT06_IN:$PORT06_OUT:$PORT07_IN:$PORT07_OUT:$PORT08_IN:$PORT08_OUT:$PORT09_IN:$PORT09_OUT:$PORT10_IN:$PORT10_OUT:$PORT11_IN:$PORT11_OUT:$PORT12_IN:$PORT12_OUT";
	RRDs::update($PORT_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $PORT_RRD: $err\n") if $err;
}

sub user {
	my $SMB = 0;
	my $MAC = 0;
	my $USR = `who -q | grep users | sed 's/^.*=//'`;
	my $rrdata = "N";

	if(stat($SMBSTATUS{$OSTYPE})) {
		if($SAMBAVER eq "2") {
			$SMB = `$SMBSTATUS{$OSTYPE} -b 2>/dev/null | wc -l`;
			$SMB -= 5;
		} elsif($SAMBAVER eq "3") {
			$SMB = `$SMBSTATUS{$OSTYPE} -b 2>/dev/null | wc -l`;
			$SMB -= 4;
		}
	}
	if(stat($MACUSERS{$OSTYPE})) {
		$MAC = `'$MACUSERS{$OSTYPE}' | wc -l`;
		$MAC--;
	}

	chomp($USR, $SMB, $MAC);
	$rrdata .= ":$USR:$SMB:$MAC";
	RRDs::update($USER_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $USER_RRD: $err\n") if $err;
}

sub nginx {
	use LWP::UserAgent;

	my $ua = LWP::UserAgent->new(timeout => 30);
	my $URL;
	my $port = "";
	my $requests = 0;
	my $total = 0;
	my $reading = 0;
	my $writing = 0;
	my $waiting = 0;
	my $NGINX_IN = 0;
	my $NGINX_OUT = 0;
	my $rrdata = "N";

	if(defined($NGINX_PORT)) {
		$port = ":" . $NGINX_PORT;
	}
	$URL = "http://127.0.0.1" . $port . "/nginx_status";
	my $response = $ua->request(HTTP::Request->new('GET', $URL));

	foreach (split(/\n/, $response->content)) {
		$total = $1 if (/^Active connections:\s+(\d+)/);
		if (/^Reading:\s+(\d+).*Writing:\s+(\d+).*Waiting:\s+(\d+)/) {
			$reading = $1;
			$writing = $2;
			$waiting = $3;
		}
		$requests = $3 if (/^\s+(\d+)\s+(\d+)\s+(\d+)/);
	}

	if($IPTABLES) {
		open(IPTABLES, "$IPTABLES -nxvL INPUT |");
		while(<IPTABLES>) {
			if(/NGINX_IN/) {
				(undef, $NGINX_IN) = split(' ', $_);
			}
		}
		close(IPTABLES);
		open(IPTABLES, "$IPTABLES -nxvL OUTPUT |");
		while(<IPTABLES>) {
			if(/NGINX_OUT/) {
				(undef, $NGINX_OUT) = split(' ', $_);
			}
		}
		close(IPTABLES);
	}

	chomp($requests, $total, $reading, $writing, $waiting, $NGINX_IN, $NGINX_OUT);
	$rrdata .= ":$requests:$total:$reading:$writing:$waiting:$NGINX_IN:$NGINX_OUT";
	RRDs::update($NGINX_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $NGINX_RRD: $err\n") if $err;
}

sub ints {
	my @INT;
	my $n;
	my $maxints;

	if(($OSTYPE eq "Linux-RHFC") ||
	  ($OSTYPE eq "Linux-Arch") ||
	  ($OSTYPE eq "Linux-Debian") ||
	  ($OSTYPE eq "Linux-Gentoo") ||
	  ($OSTYPE eq "Linux-Slack") ||
	  ($OSTYPE eq "Linux-SuSE") ||
	  ($OSTYPE eq "Linux-Generic")) {
		open(INTERRUPTS, "/proc/stat");
		while(<INTERRUPTS>) {
			if(/^intr/) {
				my @tmp = split(' ', $_);
				(undef, undef, @INT) = @tmp;
				last;
			}
		}
		close(INTERRUPTS);
	} elsif($OSTYPE eq "FreeBSD") {
		open(INTERRUPTS, "vmstat -i |");
		my @allfields;
		my $num;
		my $name;
		my $ticks;
		$maxints = 0;
		while(<INTERRUPTS>) {
			if(/^\D{3}\d+/) {
				@allfields = split(' ', $_);
				$num = $allfields[0];
				$name = $allfields[1];
				$ticks = $allfields[$#allfields - 1];
				chomp($ticks);
				if($name eq "timer") {
					$num = 0;
				} else {
					$num =~ s/^\D{3}//;
					$num =~ s/://;
				}
				$INT[$num] += $ticks;
				$maxints = $maxints < $num ? $num : $maxints;
			}
		}
		close(INTERRUPTS);
		for($n = 0; $n < $maxints; $n++) {
			$INT[$n] = !$INT[$n] ? 0 : $INT[$n];
		}
	}
	my $rrdata = "N";
	for($n = 0; $n < scalar(@INT); $n++) {
		if(($n % 256) != $n) {
			$INT[$n % 256] += $INT[$n];
		}
	}

	for($n = 0; $n < 256; $n++) {
		if(!defined($INT[$n])) {
			$INT[$n] = 0;
		}
		$rrdata .= ":" . $INT[$n];
	}
	RRDs::update($INT_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $INT_RRD: $err\n") if $err;
}

sub pc {
	my $n;
	my $ip;
	my @PC_BYTES_IN;
	my @PC_BYTES_OUT;
	my $rrdata = "N";

	for($n = 0; $n < scalar(@PC_LIST); $n++) {
		$PC_BYTES_IN[$n] = 0;
		$PC_BYTES_OUT[$n] = 0;
		if(!($ip = $PC_IP[$n])) {
			if(!(gethostbyname($PC_LIST[$n]))) {
				print "DNS problem with: ", $PC_LIST[$n], "\n";
			}
			$ip = inet_ntoa((gethostbyname($PC_LIST[$n]))[4]);
		}
		$ip=~ s/\/\d+//;
		$PC_BYTES_IN[$n] = `$IPTABLES -nxvL $PC_LIST[$n]_total | grep $ip | tail -1 | awk -F " " '{ print \$2 }'`;
		$PC_BYTES_OUT[$n] = `$IPTABLES -nxvL $PC_LIST[$n]_total | grep $ip | head -1 | awk -F " " '{ print \$2 }'`;
		chomp($PC_BYTES_IN[$n], $PC_BYTES_OUT[$n]);
		$rrdata .= ":$PC_BYTES_IN[$n]:$PC_BYTES_OUT[$n]";
	}
	for(; $n < 100; $n++) {
		$PC_BYTES_IN[$n] = 0;
		$PC_BYTES_OUT[$n] = 0;
		$rrdata .= ":$PC_BYTES_IN[$n]:$PC_BYTES_OUT[$n]";
	}
	RRDs::update($PC_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $PC_RRD: $err\n") if $err;
}

sub temp_ML310 {
	my $SB;
	my $CPU1;
	my $rrdata = "N";

	if (-e "/proc/cpqtemp") {
		$SB = `cat /proc/cpqtemp | grep "System Board" | awk -F " " '{ print \$7 }' | sed s'/C//'`;
		$CPU1 = `cat /proc/cpqtemp | grep "CPU (1)" | awk -F " " '{ print \$7 }' | sed s'/C//'`;
	} else {
		$SB = `/sbin/hplog -t | grep "System Board" | head -1 | cut -b 48- | awk -F " " '{ print \$1 }' | sed s'/C//'`;
		$CPU1 = `/sbin/hplog -t | grep "CPU (1)" | head -1 | cut -b 48- | awk -F " " '{ print \$1 }' | sed s'/C//'`;
	}

	chomp($SB, $CPU1);
	$SB = 0 unless $SB ne "---";
	$CPU1 = 0 unless $CPU1 ne "---";
	$rrdata .= ":$SB:$CPU1";
	RRDs::update($HPTEMP_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $HPTEMP_RRD: $err\n") if $err;
}

sub temp_ML350 {
	my $SB;
	my $CPU1;
	my $CPU2;
	my $rrdata = "N";

	if (-e "/proc/cpqtemp") {
		$SB = `cat /proc/cpqtemp | grep "System Board" | awk -F " " '{ print \$7 }' | sed s'/C//'`;
		$CPU1 = `cat /proc/cpqtemp | grep "CPU (1)" | awk -F " " '{ print \$7 }' | sed s'/C//'`;
		$CPU2 = `cat /proc/cpqtemp | grep "CPU (2)" | awk -F " " '{ print \$7 }' | sed s'/C//'`;
	} else {
		$SB = `/sbin/hplog -t | grep "Board" | head -1 | cut -b 48- | awk -F " " '{ print \$1 }' | sed s'/C//'`;
		$CPU1 = `/sbin/hplog -t | grep "CPU (1)" | head -1 | cut -b 48- | awk -F " " '{ print \$1 }' | sed s'/C//'`;
		$CPU2 = `/sbin/hplog -t | grep "CPU (2)" | head -1 | cut -b 48- | awk -F " " '{ print \$1 }' | sed s'/C//'`;
	}

	chomp($SB, $CPU1, $CPU2);
	$SB = 0 unless $SB ne "---";
	$CPU1 = 0 unless $CPU1 ne "---";
	$CPU2 = 0 unless $CPU2 ne "---";
	$rrdata .= ":$SB:$CPU1:$CPU2";
	RRDs::update($HPTEMP_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $HPTEMP_RRD: $err\n") if $err;
}

sub temp_ML570 {
	my $CPU1;
	my $CPU2;
	my $CPU3;
	my $CPU4;
	my $SB1;
	my $SB2;
	my $SCS1;
	my $SCS2;
	my $rrdata = "N";

	if (-e "/proc/cpqtemp") {
		$CPU1 = `cat /proc/cpqtemp | grep "CPU (1)" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$CPU2 = `cat /proc/cpqtemp | grep "CPU (2)" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$CPU3 = `cat /proc/cpqtemp | grep "CPU (4)" | tr "-" " " | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$CPU4 = `cat /proc/cpqtemp | grep "CPU (8)" | tr "-" " " | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$SB1 = `cat /proc/cpqtemp | grep "5  LM75 Sensor" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$SB2 = `cat /proc/cpqtemp | grep "6  LM75 Sensor" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$SCS1 = `cat /proc/cpqtemp | grep "7  LM75 Sensor" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$SCS2 = `cat /proc/cpqtemp | grep "8  LM75 Sensor" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
	} else {
		$CPU1 = `/sbin/hplog -t | grep "CPU (1)" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$CPU2 = `/sbin/hplog -t | grep "CPU (2)" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$CPU3 = `/sbin/hplog -t | grep "CPU (4)" | tr "-" " " | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$CPU4 = `/sbin/hplog -t | grep "CPU (8)" | tr "-" " " | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$SB1 = `/sbin/hplog -t | grep "5  LM75 Sensor" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$SB2 = `/sbin/hplog -t | grep "6  LM75 Sensor" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$SCS1 = `/sbin/hplog -t | grep "7  LM75 Sensor" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
		$SCS2 = `/sbin/hplog -t | grep "8  LM75 Sensor" | awk -F " " '{ print \$8 }' | sed s'/C.*\$//'`;
	}

	chomp($CPU1,
		$CPU2,
		$CPU3,
		$CPU4,
		$SB1,
		$SB2,
		$SCS1,
		$SCS2);
	$rrdata .= ":$CPU1:$CPU2:$CPU3:$CPU4:$SB1:$SB2:$SCS1:$SCS2";
	RRDs::update($HPTEMP_RRD, $rrdata);
	my $err = RRDs::error;
	die("ERROR: while updating $HPTEMP_RRD: $err\n") if $err;
}
